{
  This file contains both the code for the WizardModern form and the code for
  the WizardClassic form. Although using .inc files is kind of crude, this
  avoids duplicate code and major reconstruction to Inno Setup. Delphi seems to
  have no problem with this trick, everything works fine.
}

const
  {$IFDEF WIZARDCLASSIC}
    NotebookPages: array[TWizardPage, 0..1] of Integer =
      ((0, 0), (0, 1), (1, -1), (2, -1), (0, 2), (0, 3), (0, 4), (0, 5), (0, 6), (3, -1), (0, 7));

    PageCaptions: array[TWizardPage] of TSetupMessageID =
      (msgWizardWelcome, MsgWizardPassword, msgWizardLicense, msgWizardInfoBefore,
       msgWizardSelectDir, msgWizardSelectComponents, msgWizardSelectProgramGroup, msgWizardSelectTasks, msgWizardReady,
       msgWizardInfoAfter, msgWizardFinished);
  {$ELSE}
    NotebookPages: array[TWizardPage, 0..1] of Integer =
      ((0, -1), (1, 0), (1, 1), (1, 2), (1, 3), (1, 4), (1, 5), (1, 6), (1, 7), (1, 8), (1, 9), (2, -1));

    PageCaptions: array[TWizardPage] of TSetupMessageID =
      (msgWizardWelcome, MsgWizardPassword, msgWizardLicense, msgWizardInfoBefore,
       msgWizardSelectDir, msgWizardSelectComponents, msgWizardSelectProgramGroup, msgWizardSelectTasks, msgWizardReady, msgWizardInstalling,
       msgWizardInfoAfter, msgWizardFinished);
  {$ENDIF}

var

{$IFNDEF WIZARDCLASSIC}
  PageDescriptions: array[TWizardPage] of String;
{$ENDIF}

  MinimumComponentsSpace: LongInt;

function IntToKBStr(const I: LongInt): String;
begin
  Result := IntToStr((I + 1023) div 1024);
end;

function IntToMBStr (const I: Longint): String;
var
  MB, MBFrac, MBDec: Longint;
begin
  MB := I div 1048576;
  MBFrac := (I mod 1048576) * 10;
  MBDec := MBFrac div 1048576;
  if MBFrac mod 1048576 <> 0 then  { round up }
    Inc (MBDec);
  if MBDec = 10 then begin
    Inc (MB);
    MBDec := 0;
  end;
  Result := Format('%d%s%d', [MB, DecimalSeparator, MBDec]);
end;

function FixLabel (const S: String; const Space: LongInt): String;
begin
  Result := S;
  {don't localize these}
  StringChange (Result, '[name]', SetupHeader.AppName);
  StringChange (Result, '[name/ver]', SetupHeader.AppVerName);
  StringChange (Result, '[kb]', IntToKBStr(Space));
  StringChange (Result, '[mb]', IntToMBStr(Space));
{
  if MinimumKBSpaceStr = '' then
    MinimumKBSpaceStr := IntToKBStr(MinimumSpace);
  StringChange (Result, '[kb]', MinimumKBSpaceStr);
  if MinimumMBSpaceStr = '' then
    MinimumMBSpaceStr := IntToMBStr(MinimumSpace);
  StringChange (Result, '[mb]', MinimumMBSpaceStr);
}
end;

procedure TWizardForm.CalcComponentsSpace();
var
  SelectedComponents: TStringList;
  I: Integer;
begin
  MinimumComponentsSpace := SetupHeader.ExtraDiskSpaceRequired;

  SelectedComponents := TStringList.Create();
  GetSelectedComponents(SelectedComponents, False);

  //we can't simply sum component sizes because of shared files -> add file sizes
  for I := 0 to Entries[seFile].Count-1 do begin
    with PSetupFileEntry(Entries[seFile][I])^ do begin
      if ShouldProcessEntry(SelectedComponents, nil, Components, '', '') then begin
        if LocationEntry <> -1 then
          Inc(MinimumComponentsSpace, PSetupFileLocationEntry(Entries[seFileLocation][LocationEntry])^.OriginalSize)
        else
          Inc(MinimumComponentsSpace, ExternalSize)
      end;
    end;
  end;

  //don't forget to add extradiskspacerequired values
  for I := 0 to Entries[seComponent].Count-1 do
    with PSetupComponentEntry(Entries[seComponent][I])^ do
      if SelectedComponents.IndexOf(Name) <> -1 then
        Inc(MinimumComponentsSpace, ExtraDiskSpaceRequired);

  SelectedComponents.Free();

  ComponentsDiskSpaceLabel.Caption := FixLabel(SetupMessages[msgComponentsDiskSpaceMBLabel], MinimumComponentsSpace);
end;

{ TWizardForm }

constructor TWizardForm.Create (AOwner: TComponent);
{ Do all initialization of the wizard form. We're overriding Create instead of
  using the FormCreate event, because if an exception is raised in FormCreate
  it's not propogated out. }

  function TrySetDirListDirectory (const S: String): Boolean;
  begin
    Result := True;
    try
      DirList.Directory := S;
    except
      Result := False;
    end;
  end;

  procedure FindPreviousData (var AAppDir, AGroup, ASetupType: String; SelectedComponents, DeselectedComponents, SelectedTasks, DeselectedTasks: TStringList);
  const
    RootKeys: array[0..1] of HKEY = (HKEY_CURRENT_USER, HKEY_LOCAL_MACHINE);
  var
    I: Integer;
    H: HKEY;
    S: String; 
  begin
    AAppDir := '';
    AGroup := '';
    ASetupType := ''; 
    SelectedComponents.Clear(); 
    DeSelectedComponents.Clear(); 

    { First look in HKEY_CURRENT_USER. That's where Inno Setup creates
      the uninstall key for non-administrators. If the key doesn't exist
      under HKEY_CURRENT_USER, check HKEY_LOCAL_MACHINE. }
    for I := 0 to 1 do
      if RegOpenKeyEx(RootKeys[I],
         PChar(Format('%s\%s_is1', [NEWREGSTR_PATH_UNINSTALL, UninstallRegKeyBaseName])),
         0, KEY_QUERY_VALUE, H) = ERROR_SUCCESS then begin
        try
          { do not localize or change the following strings }
          RegQueryStringValue (H, 'Inno Setup: App Path', AAppDir);
          RegQueryStringValue (H, 'Inno Setup: Icon Group', AGroup);
          RegQueryStringValue (H, 'Inno Setup: Setup Type', ASetupType);
          if RegQueryStringValue (H, 'Inno Setup: Selected Components', S) then
            SelectedComponents.CommaText := S;
          if RegQueryStringValue (H, 'Inno Setup: Deselected Components', S) then
            DeselectedComponents.CommaText := S;
          if RegQueryStringValue (H, 'Inno Setup: Selected Tasks', S) then
            SelectedTasks.CommaText := S;
          if RegQueryStringValue (H, 'Inno Setup: Deselected Tasks', S) then
            DeselectedTasks.CommaText := S;
        finally
          RegCloseKey (H);
        end;
        Break;
      end;
  end;

var
  OldTextHeight, NewTextHeight, X, W1, W2, W3: Integer;
  S: String;
  SystemMenu: HMENU;
  PrevAppDir, PrevGroup, P, P2, P3: String;
  PrevSetupType: String;
  I, PrevSetupTypeIndex: Integer;
  TypeEntry: PSetupTypeEntry;
  ComponentEntry: PSetupComponentEntry;
  MaxComponentSize: Integer;
const
  idDirectories: array[Boolean] of TSetupMessageID =
    (msgDirectoryOld, msgDirectoryNew);
  idProgramManagers: array[Boolean] of TSetupMessageID =
    (msgProgramManagerOld, msgProgramManagerNew);
  WizardButtonWidth = 95;
begin
  inherited;

  InitialSelectedComponents := TStringList.Create();
  PrevSelectedComponents := TStringList.Create();
  PrevDeselectedComponents := TStringList.Create();
  PrevSelectedTasks := TStringList.Create();
  PrevDeselectedTasks := TStringList.Create();
  TaskLabels := TList.Create();
  TaskButtons := TList.Create();

  SetFormFont (Self, OldTextHeight, NewTextHeight);
  {$IFNDEF WIZARDCLASSIC}
    SetFontNameSize (WelcomeLabel1.Font, SetupMessages[msg_WelcomeFont], 'Arial', 12);
    WelcomeLabel1.Font.Style := [fsBold]; 
    PageNameLabel.Font.Style := [fsBold]; 

    if shWindowVisible in SetupHeader.Options then 
      Caption := SetupMessages[msgSetupAppTitle]
    else
      Caption := FmtSetupMessage1(msgSetupWindowTitle, SetupHeader.AppName);
  {$ENDIF}

  { Give it a minimize button if main window isn't visible }
  if not(shWindowVisible in SetupHeader.Options) then begin
    BorderIcons := BorderIcons + [biMinimize];
    BorderStyle := bsSingle;
  end;

  { Position the buttons, and scale their size }
  W1 := MulDiv(WizardButtonWidth, NewTextHeight, OldTextHeight);  { width of each button }
  {$IFDEF WIZARDCLASSIC}
    W2 := MulDiv(8, NewTextHeight, OldTextHeight); 
  {$ELSE}
    W2 := MulDiv(12, NewTextHeight, OldTextHeight); 
  {$ENDIF}
  W3 := MulDiv(10, NewTextHeight, OldTextHeight);  { space between Next and Cancel button }

  BackButton.Width := W1;
  NextButton.Width := W1;
  CancelButton.Width := W1;
  X := ClientWidth - W2 - W1;
  CancelButton.Left := X;
  Dec (X, W3);
  Dec (X, W1);
  NextButton.Left := X;
  Dec (X, W1);
  BackButton.Left := X;

  {$IFDEF WIZARDCLASSIC}
    WelcomeLabel.Caption := FixLabel(SetupMessages[msgWelcomeLabel1] + ' ' + SetupMessages[msgWelcomeLabel2], MinimumSpace) +
      SNewLine2 + SetupMessages[msgClickNext];
    ClickLabel.Caption := SetupMessages[msgClickNext];
    PasswordLabel.Caption := SetupMessages[msgPasswordLabel1] + ' ' + SetupMessages[msgPasswordLabel2];
    InfoBeforeLabel.Caption := SetupMessages[msgInfoBeforeLabel];
    SelectFolderLabel.Caption := FixLabel(FmtSetupMessage1(msgSelectFolderLabel,
      SetupMessages[idDirectories[NewGUI]]), MinimumSpace);
    SelectComponentsLabel.Caption := FixLabel(SetupMessages[msgSelectComponentsLabel], MinimumSpace);
    IconsLabel.Caption := FmtSetupMessage1(msgIconsLabel,
      SetupMessages[idProgramManagers[NewGUI]]);
    SelectTasksLabel.Caption := FixLabel(SetupMessages[msgSelectTasksLabel], MinimumSpace);
    S := FixLabel(SetupMessages[msgReadyLabel1], MinimumSpace) + SNewLine2;
    if not(shDisableDirPage in SetupHeader.Options) or HasComponents or (HasTasklessIcons and not(shDisableProgramGroupPage in SetupHeader.Options)) or HasTasks then
      S := S + SetupMessages[msgReadyLabel2a]
    else
      S := S + SetupMessages[msgReadyLabel2b];
    ChangeReadyLabel (S);
    InfoAfterLabel.Caption := SetupMessages[msgInfoAfterLabel];
    InfoAfterClickLabel.Caption := SetupMessages[msgInfoAfterClickLabel];
  {$ELSE}
    S := FixLabel(SetupMessages[msgWelcomeLabel1], MinimumSpace);
    ChangeWelcomeLabel1(S);
    WelcomeLabel2.Caption := FixLabel(SetupMessages[msgWelcomeLabel2], MinimumSpace) +
      SNewLine2 + SetupMessages[msgClickNextModern];
    PageDescriptions[wpPassword] := SetupMessages[msgPasswordLabel1];
    PasswordLabel.Caption := SetupMessages[msgPasswordLabel2];
    PasswordClickLabel.Caption := SetupMessages[msgClickNextModern];
    PageDescriptions[wpLicense] := SetupMessages[msgLicenseLabel];
    PageDescriptions[wpInfoBefore] := SetupMessages[msgInfoBeforeLabel]; 
    PageDescriptions[wpSelectDir] := FixLabel(FmtSetupMessage1(msgSelectFolderLabel, SetupMessages[idDirectories[NewGUI]]), MinimumSpace);
    SelectDirClickLabel.Caption := SetupMessages[msgClickNextModern];
    PageDescriptions[wpSelectComponents] := FixLabel(SetupMessages[msgSelectComponentsLabel], MinimumSpace);
    SelectComponentsClickLabel.Caption := SetupMessages[msgClickNextModern];
    PageDescriptions[wpSelectProgramGroup] := FmtSetupMessage1(msgIconsLabel, SetupMessages[idProgramManagers[NewGUI]]);
    SelectProgramGroupClickLabel.Caption := SetupMessages[msgClickNextModern];
    if not(shAllowNoIcons in SetupHeader.Options) then
      GroupList.Height := ComponentsList.Height;
    PageDescriptions[wpSelectTasks] := FixLabel(SetupMessages[msgSelectTasksLabel], MinimumSpace);
    SelectTasksClickLabel.Caption := SetupMessages[msgClickNextModern];
    PageDescriptions[wpReady] := FixLabel(SetupMessages[msgReadyLabel1], MinimumSpace);
    if not(shDisableDirPage in SetupHeader.Options) or HasComponents or (HasTasklessIcons and not(shDisableProgramGroupPage in SetupHeader.Options)) or HasTasks then
      S := SetupMessages[msgReadyLabel2a]
    else
      S := SetupMessages[msgReadyLabel2b];
    ChangeReadyLabel (S);
    PageDescriptions[wpInstalling] := FixLabel(SetupMessages[msgInstallingLabel], MinimumSpace);
    InstallingPanel.Caption := '';
    PageDescriptions[wpInfoAfter] := SetupMessages[msgInfoAfterLabel];
    InfoAfterClickLabel.Caption := SetupMessages[msgInfoAfterClickLabel];
  {$ENDIF}

  BeveledLabel.Caption := ' ' + SetupMessages[msgBeveledLabel];
  PasswordEditLabel.Caption := SetupMessages[msgPasswordEditLabel];
  LicenseLabel1.Caption := FixLabel(SetupMessages[msgLicenseLabel1], MinimumSpace);
  LicenseLabel2.Caption := FixLabel(SetupMessages[msgLicenseLabel2], MinimumSpace);
  InfoBeforeClickLabel.Caption := SetupMessages[msgInfoBeforeClickLabel];
  DiskSpaceLabel.Caption := FixLabel(SetupMessages[msgDiskSpaceMBLabel], MinimumSpace);
  ComponentsDiskSpaceLabel.Caption := FixLabel(SetupMessages[msgComponentsDiskSpaceMBLabel], MinimumSpace);

  if HasTasklessIcons then
    NoIconsCheck.Caption := FmtSetupMessage1(msgNoProgramGroupCheck, SetupMessages[idProgramManagers[NewGUI]])
  else
    NoIconsCheck.Caption := SetupMessages[msgNoIconsCheck];

  if HasIcons then
    S := FixLabel(SetupMessages[msgFinishedLabel], MinimumSpace)
  else
    S := FixLabel(SetupMessages[msgFinishedLabelNoIcons], MinimumSpace);
  OrigFinishedLabelWidth := FinishedLabel.Width;
  ChangeFinishedLabel (S + SNewLine2 + SetupMessages[msgClickFinish]);
  FirstRunButton := nil;
  YesRadio.Caption := SetupMessages[msgYesRadio];
  NoRadio.Caption := SetupMessages[msgNoRadio];

  { Don't set UseRichEdit to True on the TRichEditViewers unless they are going
    to be used. There's no need to load riched*.dll unnecessarily. }
  if SetupHeader.LicenseText <> '' then begin
    LicenseMemo.UseRichEdit := True;
    LicenseMemo.RTFText := SetupHeader.LicenseText;
  end;
  if SetupHeader.InfoBeforeText <> '' then begin
    InfoBeforeMemo.UseRichEdit := True;
    InfoBeforeMemo.RTFText := SetupHeader.InfoBeforeText;
  end;
  { Note: InfoAfterMemo is set up in the Main unit }

  { Center the form inside the MainForm's client area }
  MainForm.CenterForm (Self);

  { Append an 'About Setup' item to the wizard form also }
  SystemMenu := GetSystemMenu(Handle, False);
  AppendMenu (SystemMenu, MF_SEPARATOR, 0, nil);
  AppendMenu (SystemMenu, MF_STRING, 9999, PChar(SetupMessages[msgAboutSetupMenuItem]));

  { Read settings from a previous install if available }
  FindPreviousData (PrevAppDir, PrevGroup, PrevSetupType, PrevSelectedComponents, PrevDeselectedComponents, PrevSelectedTasks, PrevDeselectedTasks);
  PrevSetupTypeIndex := -1; //assigned later

  { Assign default directory name }
  if shCreateAppDir in SetupHeader.Options then begin
    { Try different values and see what works. The 'except end;' throws away any
      exceptions that might occur. }
    TrySetDirListDirectory ('\');
    TrySetDirListDirectory (SystemDrive + '\');
    P := '';
    if shUsePreviousAppDir in SetupHeader.Options then begin
      P := PrevAppDir;
      PreviousAppDir := PrevAppDir;
    end;
    if P = '' then
      P := ChangeDirConst(SetupHeader.DefaultDirName);
    if not(shDisableAppendDir in SetupHeader.Options) or
       not TrySetDirListDirectory(P) then begin
      P2 := P;
      while True do begin
        P3 := RemoveBackslashUnlessRoot(ExtractFilePath(P2));
        if Length(P3) < 3 then Break;
        if TrySetDirListDirectory(P3) or (P3 = P2) then
          Break;
        P2 := P3;
      end;
    end;
    DirEdit.Text := P;

    if (InitDir <> '') and not(shDisableDirPage in SetupHeader.Options) then begin
      { ^ InitDir currently isn't supported for installations with
        shDisableDirPage set. If the wizard page isn't displayed, it doesn't
        get a chance to validate the path specified. }
      try
        DirList.Directory := ExtractFilePath(InitDir);
        DirList.Directory := InitDir;
      except
      end;
      DirEdit.Text := InitDir;
    end;
  end
  else
    DirEdit.Text := WinDir;

  { Fill types list and assign default type } 
  if Entries[seType].Count > 0 then begin
    //first fill list
    TypesCombo.Clear();
    for I := 0 to Entries[seType].Count-1 do begin
      TypeEntry := PSetupTypeEntry(Entries[seType][I]);
      TypesCombo.Items.AddObject(TypeEntry.Description, TObject(TypeEntry));
      if Comparetext(TypeEntry.Name, PrevSetupType) = 0 then
        PrevSetupTypeIndex := I;
    end;
    //now assign default type
    if (PrevSetupTypeIndex <> -1) and (shUsePreviousSetupType in SetupHeader.Options) then
      TypesCombo.ItemIndex := PrevSetupTypeIndex
    else
      TypesCombo.ItemIndex := 0;
  end;

  { Fill components list and assign default components}
  //find the biggest component
  MaxComponentSize := 0;
  for I := 0 to Entries[seComponent].Count-1 do begin
    ComponentEntry := PSetupComponentEntry(Entries[seComponent][I]);
    if ComponentEntry.Size > MaxComponentSize then
      MaxComponentSize := ComponentEntry.Size;
  end;

  //first fill list
  ComponentsList.Clear();
  ComponentsList.Flat := shFlatComponentsList in SetupHeader.Options;
  for I := 0 to Entries[seComponent].Count-1 do begin
    ComponentEntry := PSetupComponentEntry(Entries[seComponent][I]);
    ComponentsList.Items.Add(ComponentEntry.Description);
    ComponentsList.ItemObject[I] := TObject(ComponentEntry);
    ComponentsList.ItemEnabled[I] := not (coFixed in ComponentEntry.Options);
    if (shShowComponentSizes in SetupHeader.Options) then begin
      if MaxComponentSize < 1024*1024 then
        ComponentsList.ItemSubItem[I] := FmtSetupMessage1(msgComponentSize1, IntToKBStr(ComponentEntry.Size))
      else
        ComponentsList.ItemSubItem[I] := FmtSetupMessage1(msgComponentSize2, IntToMBStr(ComponentEntry.Size));
    end;
  end;

  //now assign default type
  if InitComponents.Count <> 0 then begin
    SetSelectedComponents(InitComponents, nil);
    //try to find a custom type. if there's no custom type the UsePreviousSetupType won't
    //work and if the install is not silent the type list won't display a custom type.
    if HasCustomType then begin
      for I := 0 to Entries[seType].Count-1 do begin
        TypeEntry := PSetupTypeEntry(Entries[seType][I]);
        if toIsCustom in TypeEntry.Options then begin
          TypesCombo.ItemIndex := I;
          Break;
        end;
      end;
    end;
  end else begin
    if (PrevSetupTypeIndex <> -1) and (shUsePreviousSetupType in SetupHeader.Options) then begin
      TypeEntry := PSetupTypeEntry(Entries[seType][PrevSetupTypeIndex]);
      if toIsCustom in TypeEntry.Options then begin
        //the previous setup type is a custom type: first select the default components
        //for the default type (usually the full type). needed for new components.
        TypeEntry := PSetupTypeEntry(Entries[seType][0]);
        SetSelectedComponentsFromType(TypeEntry.Name, False);
        //now restore the customization
        SetSelectedComponents(PrevSelectedComponents, PrevDeselectedComponents);
      end else begin
        //this is not a custom type, so just select components based on the previous type
        TypeEntry := PSetupTypeEntry(Entries[seType][PrevSetupTypeIndex]);
        SetSelectedComponentsFromType(TypeEntry.Name, False);
      end;
    end else if Entries[seType].Count > 0 then begin
      TypeEntry := PSetupTypeEntry(Entries[seType][0]);
      SetSelectedComponentsFromType(TypeEntry.Name, False);
    end;
  end;

  CalcComponentsSpace();

  //Show or hide the components list based on the selected type
  if HasCustomType then begin
    TypeEntry := PSetupTypeEntry(Entries[seType][TypesCombo.ItemIndex]);
    if (toIsCustom in TypeEntry.Options) or (shAlwaysShowComponentsList in SetupHeader.Options) then
      ComponentsList.Visible := True
    else
      ComponentsList.Visible := False;
  end else
    ComponentsList.Visible := False;
  ComponentsDiskSpaceLabel.Visible := ComponentsList.Visible;

  //Store the initial setup type and components (only necessary if customizable)
  if HasCustomType then begin
    InitialSetupTypeIndex := TypesCombo.ItemIndex;
    GetSelectedComponents(InitialSelectedComponents, False);
  end;

  { Assign default group name }
  if (InitProgramGroup <> '') and not(shDisableProgramGroupPage in SetupHeader.Options) then
    { ^ InitProgramGroup currently isn't supported for installations with
      shDisableProgramGroupPage set. If the wizard page isn't displayed, it
      doesn't get a chance to validate the program group name specified. }
    GroupEdit.Text := InitProgramGroup
  else begin
    if (PrevGroup = '') or not(shUsePreviousGroup in SetupHeader.Options) then
      GroupEdit.Text := SetupHeader.DefaultGroupName
    else
      GroupEdit.Text := PrevGroup;
  end;

  if shAllowNoIcons in SetupHeader.Options then begin
    if InitNoIcons then
      NoIconsCheck.Checked := True;
    NoIconsCheck.Visible := True;
  end else
    NoIconsCheck.Visible := False;

  if SetupHeader.InstallMode = imNormal then begin
    //tasks page will be updated later based on the selected components
    NeedSelectTasksPageUpdate := True;
  end else begin
    //update tasks page now
    UpdateSelectTasksPage();
    NeedSelectTasksPageUpdate := False; //not really necessary
  end;

  ReadyMemo.Visible := not (shDisableReadyMemo in SetupHeader.Options);

  CurPage := Low(TWizardPage);
  CurPageChanged;

  ReadGroupList;
end;

procedure TWizardForm.FormDestroy(Sender: TObject);
begin
  TaskButtons.Free();
  TaskLabels.Free();
  PrevDeselectedComponents.Free();
  PrevSelectedTasks.Free();
  PrevDeselectedTasks.Free();
  PrevSelectedComponents.Free();
  InitialSelectedComponents.Free();
end;

{$IFNDEF WIZARDCLASSIC}
procedure TWizardForm.ChangeWelcomeLabel1 (const S: String);
var
  Width, Y, dY: Integer;
begin
  Width := WelcomeLabel1.Width;
  WelcomeLabel1.Caption := S + SNewLine;
  WelcomeLabel1.AutoSize := True;
  WelcomeLabel1.Width := Width;
  Y := WelcomeLabel1.Top + WelcomeLabel1.Height;
  dY := Y - WelcomeLabel2.Top;
  WelcomeLabel2.Top := Y;
  WelcomeLabel2.Height := WelcomeLabel2.Height - dY;
end;
{$ENDIF}

procedure TWizardForm.ChangeReadyLabel (const S: String);
var
  Width, OldHeight, dY: Integer;
begin
  Width := ReadyLabel.Width;
  OldHeight := ReadyLabel.Height;
  ReadyLabel.Caption := S;
  ReadyLabel.AutoSize := True;
  ReadyLabel.Width := Width;
  dY := ReadyLabel.Height-OldHeight;
  ReadyMemo.Top := ReadyMemo.Top+dY;
  ReadyMemo.Height := ReadyMemo.Height-dY;
end;

procedure TWizardForm.ChangeFinishedLabel (const S: String);
var
  Y: Integer;
begin
  FinishedLabel.AutoSize := False;
  FinishedLabel.Width := OrigFinishedLabelWidth;
  FinishedLabel.Caption := S + SNewLine;
  FinishedLabel.AutoSize := True;
  Y := FinishedLabel.Top + FinishedLabel.Height;
  //ShowReadmeCheck.Top := Y;
  YesRadio.Top := Y;
  NoRadio.Top := Y + MulDiv(24, Screen.PixelsPerInch, 96);
end;

function CreateLabel(const BaseControl: TControl; const ACaption: String; var Y: Integer): TLabel;
var
  ALabel: TLabel;
begin
  ALabel := TLabel.Create(BaseControl.Owner);
  with ALabel do begin
    Parent := BaseControl.Parent;
    AutoSize := False;
    Caption := ACaption;
    Left := BaseControl.Left;
    Top := Y;
    Width := BaseControl.Width;
    Height := BaseControl.Height;
    Y := Y + MulDiv(20, Screen.PixelsPerInch, 96);
  end;

  Result := ALabel;
end;

function CreateCheckBox(const BaseControl: TControl; const ACaption: String; const AChecked: Boolean; const ATag: Integer; var Y: Integer): TButtonControl;
var
  ACheckBox: TCheckBox;
begin
  ACheckBox := TCheckBox.Create(BaseControl.Owner);
  with ACheckBox do begin
    Parent := BaseControl.Parent;
    Caption := ACaption;
    Left := BaseControl.Left;
    Top := Y;
    Width := BaseControl.Width;
    Height := BaseControl.Height;
    Checked := AChecked;
    Tag := ATag;
    Y := Y + MulDiv(24, Screen.PixelsPerInch, 96);
  end;

  Result := ACheckBox;
end;

function CreateRadioButton(const BaseControl: TControl; const ACaption, AGroup: String; const AChecked: Boolean; const ATag: Integer; var Y: Integer): TButtonControl;
var
  ARadioButton: TGroupRadioButton;
begin
  ARadioButton := TGroupRadioButton.Create(BaseControl.Owner);
  with ARadioButton do begin
    Parent := BaseControl.Parent;
    Caption := ACaption;
    Group := AGroup;
    Left := BaseControl.Left;
    Top := Y;
    Width := BaseControl.Width;
    Height := BaseControl.Height;
    Checked := AChecked;
    Tag := ATag;
    Y := Y + MulDiv(24, Screen.PixelsPerInch, 96);
  end;

  Result := ARadioButton;
end;

procedure TWizardForm.CreateRunButtons(const SelectedComponents, SelectedTasks: TStringList; RunButtons: TList);
var
  RunEntry: PSetupRunEntry;
  RunButton: TButtonControl;
  Caption: String;
  I, Y: Integer;
begin
  Y := YesRadio.Top;

  for I := 0 to Entries[seRun].Count-1 do begin
    RunEntry := PSetupRunEntry(Entries[seRun][I]);
    if ShouldProcessRunEntry(SelectedComponents, SelectedTasks, RunEntry) and (roPostInstall in RunEntry.Options) then begin
      if RunEntry.Description <> '' then
        Caption := RunEntry.Description
      else if not(roShellExec in RunEntry.Options) then
        Caption := FmtSetupMessage1(msgRunEntryExec, ExtractFileName(RunEntry.Name))
      else
        Caption := FmtSetupMessage1(msgRunEntryShellExec, ExtractFileName(RunEntry.Name));
      RunButton := CreateCheckBox(YesRadio, Caption, not(roUnchecked in RunEntry.Options), Integer(RunEntry), Y);
      RunCheckBoxes.Add(RunButton);
      if FirstRunButton = nil then
        FirstRunButton := RunButton;
    end;
  end;
end;

procedure TWizardForm.CreateTaskButtons(const SelectedComponents: TStringList);

  function MultipleRadioButtonsInGroup(const SelectedComponents: TStringList; const GroupDescription: String): Boolean;
  var
    TaskEntry: PSetupTaskEntry;
    N, I: Integer;
  begin
    N := 0;
    for I := 0 to Entries[seTask].Count-1 do begin
      TaskEntry := PSetupTaskEntry(Entries[seTask][I]);
      if ShouldProcessEntry(SelectedComponents, nil, TaskEntry.Components, '', '') and (TaskEntry.GroupDescription = GroupDescription) and (toExclusive in TaskEntry.Options) then begin
        Inc(N);
        if N > 1 then begin
          Result := True;
          Exit;
        end;
      end;
    end;
    Result := False;
  end;

var
  TaskEntry: PSetupTaskEntry;
  TaskLabel: TLabel;
  TaskButton: TButtonControl;
  I, P, Y: Integer;
  LastGroupDescription: String;
  UseRadioButtons: Boolean;
begin
  for I := 0 to TaskLabels.Count-1 do
    TLabel(TaskLabels[I]).Free();
  TaskLabels.Clear();

  for I := 0 to TaskButtons.Count-1 do
    TButtonControl(TaskButtons[I]).Free();
  TaskButtons.Clear();

  Y := BaseTaskLabel.Top;
  UseRadioButtons := MultipleRadioButtonsInGroup(SelectedComponents, '');
  LastGroupDescription := '';

  //Create the task buttons with their default checked states
  for I := 0 to Entries[seTask].Count-1 do begin
    TaskEntry := PSetupTaskEntry(Entries[seTask][I]);
    if ShouldProcessEntry(SelectedComponents, nil, TaskEntry.Components, '', '') then begin
      //see if we should add a group label
      if TaskEntry.GroupDescription <> LastGroupDescription then begin
        TaskLabel := CreateLabel(BaseTaskLabel, TaskEntry.GroupDescription, Y);
        TaskLabels.Add(TaskLabel);
        UseRadioButtons := MultipleRadioButtonsInGroup(SelectedComponents, TaskEntry.GroupDescription);
        LastGroupDescription := TaskEntry.GroupDescription;
      end;

      //create a checkbox or radiobutton. only create a radiobutton if the group
      //has multiple radiobuttons based on the selected components
      if UseRadioButtons and (toExclusive in TaskEntry.Options) then
        TaskButton := CreateRadioButton(BaseTaskCheckBox, TaskEntry.Description, TaskEntry.GroupDescription, not(toUnchecked in TaskEntry.Options), Integer(TaskEntry), Y)
      else
        TaskButton := CreateCheckBox(BaseTaskCheckBox, TaskEntry.Description, not(toUnchecked in TaskEntry.Options), Integer(TaskEntry), Y);

      TaskButtons.Add(TaskButton);
    end;
  end;

  //Now restore the previous checked state of the buttons we just created
  if shUsePreviousTasks in SetupHeader.Options then begin
    for I := 0 to TaskButtons.Count-1 do begin
      TaskButton := TaskButtons[I];
      TaskEntry := PSetupTaskEntry(TaskButton.Tag);

      P := PrevSelectedTasks.IndexOf(TaskEntry.Name);
      if P >= 0 then begin
        if TaskButton is TCheckBox then
          TCheckBox(TaskButton).Checked := True
        else
          TGroupRadioButton(TaskButton).Checked := True;
        Continue;
      end;

      P := PrevDeselectedTasks.IndexOf(TaskEntry.Name);
      if P >= 0 then begin
        if TaskButton is TCheckBox then
          TCheckBox(TaskButton).Checked := False
        else
          TGroupRadioButton(TaskButton).Checked := False;
      end;
    end;
  end;
end;

function TWizardForm.GetSetupType(): PSetupTypeEntry;
var
  Index: Integer;
begin
  Index := TypesCombo.ItemIndex;
  if Index <> -1 then
    Result := PSetupTypeEntry(TypesCombo.Items.Objects[TypesCombo.ItemIndex])
  else
    Result := nil;
end;

procedure TWizardForm.SetSelectedComponents(const SelectedComponents, DeselectedComponents: TStringList);
var
  I, P: Integer;
  ComponentEntry: PSetupComponentEntry;
begin
  for I := 0 to Entries[seComponent].Count-1 do begin
    ComponentEntry := PSetupComponentEntry(Entries[seComponent][I]);

    if SelectedComponents <> nil then begin
      P := SelectedComponents.IndexOf(ComponentEntry.Name);
      if P >= 0 then begin
        ComponentsList.Checked[I] := True;
        Continue;
      end;
    end;

    if DeselectedComponents <> nil then begin
      P := DeselectedComponents.IndexOf(ComponentEntry.Name);
      if P >= 0 then
        ComponentsList.Checked[I] := False
    end;
  end;
end;

procedure TWizardForm.SetSelectedComponentsFromType(const TypeName: String; OnlySelectFixedComponents: Boolean);
var
  ComponentTypes: TStringList;
  ComponentEntry: PSetupComponentEntry;
  I: Integer;
begin
  ComponentTypes := TStringList.Create();
  for I := 0 to Entries[seComponent].Count-1 do begin
    ComponentEntry := PSetupComponentEntry(Entries[seComponent][I]);
    if not OnlySelectFixedComponents or (coFixed in ComponentEntry.Options) then begin
      ComponentTypes.CommaText := ComponentEntry.Types;
      ComponentsList.Checked[I] := ComponentTypes.IndexOf(TypeName) <> -1;
    end;
  end;
  ComponentTypes.Free();
end;

procedure TWizardForm.UpdateSelectTasksPage;
var
  SelectedComponents: TStringList;
begin
  SelectedComponents := TStringList.Create();
  GetSelectedComponents(SelectedComponents, False);
  CreateTaskButtons(SelectedComponents);
  SelectedComponents.Free();
end;

procedure TWizardForm.GetSelectedComponents(Components: TStringList; const Descriptions: Boolean);

  function GetString(ComponentEntry: PSetupComponentEntry; Descriptions: Boolean): String;
  begin
    if Descriptions then
      Result := ComponentEntry.Description
    else
      Result := ComponentEntry.Name;
  end;

var
  ComponentEntry: PSetupComponentEntry;
  I: Integer;
begin
  Components.Clear();
  for I := 0 to ComponentsList.Items.Count-1 do begin
    if ComponentsList.Checked[I] then begin
      ComponentEntry := PSetupComponentEntry(ComponentsList.ItemObject[I]);
      Components.Add(GetString(ComponentEntry, Descriptions));
    end;
  end;
end;

procedure TWizardForm.GetSelectedTasks(Tasks: TStringList; const Descriptions: Boolean);

  function GetString(TaskEntry: PSetupTaskEntry; Descriptions: Boolean): String;
  var
    I, N: Integer;
  begin
    with TaskEntry^ do begin
      if Descriptions then begin
        N := Length(Description);
        Result := '';
        for I := 1 to N do begin
          if (I = N) or not ((Description[I] = '&') and (Description[I+1] <> '&')) then
            Result := Result + Description[I];
        end;
      end else
        Result := Name;
    end;
  end;

var
  TaskEntry: PSetupTaskEntry;
  TaskButton: TButtonControl;
  Checked: Boolean;
  I: Integer;
begin
  Tasks.Clear();
  for I := 0 to TaskButtons.Count-1 do begin
    TaskButton := TaskButtons[I];
    if TaskButton is TCheckBox then
      Checked := TCheckBox(TaskButtons[I]).Checked
    else
      Checked := TGroupRadioButton(TaskButtons[I]).Checked;
    if Checked then begin
      TaskEntry := PSetupTaskEntry(TaskButton.Tag);
      Tasks.Add(GetString(TaskEntry, Descriptions));
    end;
  end;
end;

function TWizardForm.GetFirstTaskButton(): TButtonControl;
var
  TaskButton: TButtonControl;
  I: Integer;
begin
  for I := 0 to TaskButtons.Count-1 do begin
    TaskButton := TaskButtons[I];
    if not(TaskButton is TGroupRadioButton) or TGroupRadioButton(TaskButton).Checked then begin
      Result := TaskButton;
      Exit;
    end;
  end;
  Result := nil;
end;

procedure TWizardForm.CurPageChanged;
{ Call this whenever the current page is changed }
var
  FirstTaskButton: TButtonControl;
  V: Boolean;
label 1;
begin
  Notebook1.PageIndex := NotebookPages[CurPage, 0];
  if NotebookPages[CurPage, 1] <> -1 then
    Notebook2.PageIndex := NotebookPages[CurPage, 1];

  { Set button visibility and captions }
  {$IFDEF WIZARDCLASSIC}
    V := not(CurPage in [Low(TWizardPage), wpInfoAfter, wpFinished]);
  {$ELSE}
    V := not(CurPage in [Low(TWizardPage), wpInstalling, wpInfoAfter, wpFinished]);
  {$ENDIF}
  if (SetupHeader.InfoAfterText <> '') and (CurPage = wpFinished) then
    V := True;

  BackButton.Visible := V;
  {$IFNDEF WIZARDCLASSIC}
    NextButton.Visible := CurPage <> wpInstalling;
  {$ENDIF}
  CancelButton.Visible := CurPage < wpInfoAfter;

  BackButton.Caption := SetupMessages[msgButtonBack];
  case CurPage of
    wpLicense: begin
        NextButton.Caption := SetupMessages[msgButtonYes];
        CancelButton.Caption := SetupMessages[msgButtonNo];
      end;
    wpReady: begin
        NextButton.Caption := SetupMessages[msgButtonInstall];
        CancelButton.Caption := SetupMessages[msgButtonCancel];
      end;
    wpFinished: begin
        NextButton.Caption := SetupMessages[msgButtonFinish];
        CancelButton.Caption := SetupMessages[msgButtonCancel];
      end;
  else
  1:NextButton.Caption := SetupMessages[msgButtonNext];
    CancelButton.Caption := SetupMessages[msgButtonCancel];
  end;

  { Set the Caption to match the current page's title }
  {$IFDEF WIZARDCLASSIC}
    Caption := SetupMessages[PageCaptions[CurPage]];
    { Hide ClickLabel if it's on the Welcome, Ready to Install, or Setup
      Completed wizard pages }
    ClickLabel.Visible := not(CurPage in [wpWelcome, wpReady, wpFinished]);
  {$ELSE}
    PageNameLabel.Caption := SetupMessages[PageCaptions[CurPage]];
    PageDescriptionLabel.Caption := PageDescriptions[CurPage];
    if CurPage in [wpWelcome, wpFinished] then
      Notebook1.Color := clWindow
    else
      Notebook1.Color := clBtnFace;
  {$ENDIF}

  {$IFDEF WIZARDCLASSIC}
    BeveledLabel.Visible := (SetupMessages[msgBeveledLabel] <> '') and not(CurPage in [wpLicense, wpInfoBefore, wpInfoAfter]);
  {$ELSE}
    BeveledLabel.Visible := (SetupMessages[msgBeveledLabel] <> '') and not(CurPage in [wpWelcome, wpFinished]);
  {$ENDIF}

  { Don't make the Next button "default" (i.e. selectable with the Enter key)
    when on the License page }
  NextButton.Default := CurPage <> wpLicense;
  { Adjust focus }
  case CurPage of
    wpWelcome: ActiveControl := NextButton;
    wpPassword: ActiveControl := PasswordEdit;
    wpLicense: ActiveControl := LicenseMemo;
    wpInfoBefore: ActiveControl := InfoBeforeMemo;
    wpSelectDir: ActiveControl := DirEdit;
    wpSelectComponents:  ActiveControl := TypesCombo;
    wpSelectProgramGroup: begin
        if not NoIconsCheck.Checked then
          ActiveControl := GroupEdit
        else
          ActiveControl := NoIconsCheck;
      end;
    wpSelectTasks: begin
        FirstTaskButton := GetFirstTaskButton();
        if FirstTaskButton <> nil then
          ActiveControl := FirstTaskButton
      end;
    wpReady: ActiveControl := NextButton;
    {$IFNDEF WIZARDCLASSIC}
      wpInstalling: ActiveControl := CancelButton;
    {$ENDIF}
    wpInfoAfter: ActiveControl := InfoAfterMemo;
    wpFinished: begin
        if (FirstRunButton <> nil) and FirstRunButton.Visible then
          ActiveControl := FirstRunButton
        else
        if YesRadio.Visible and YesRadio.Checked then
          ActiveControl := YesRadio
        else
        if NoRadio.Visible and NoRadio.Checked then
          ActiveControl := NoRadio
        else
          ActiveControl := NextButton;
      end;
  end;

  if HasCodeText then
    ScriptRunProcedure('CurPageChanged', ['CurPage', Integer(CurPage)], False);
end;

function TWizardForm.SkipCurPage: Boolean;

  function SkipSelectTasksPage: Boolean;
  var
    SelectedComponents: TStringList;
    I: Integer;
  begin
    //check if there is a task that has to be processed based on the selected
    //components
    SelectedComponents := TStringList.Create();
    GetSelectedComponents(SelectedComponents, False);
    Result := True;
    for I := 0 to Entries[seTask].Count-1 do begin
      with PSetupTaskEntry(Entries[seTask][I])^ do begin
        if ShouldProcessEntry(SelectedComponents, nil, Components, '', '') then begin
          Result := False;
          Break;
        end;
      end;
    end;
    SelectedComponents.Free();
  end;

begin
  Result :=
    ((CurPage = wpPassword) and not NeedPassword) or
    ((CurPage = wpLicense) and (SetupHeader.LicenseText = '')) or
    ((CurPage = wpInfoBefore) and (SetupHeader.InfoBeforeText = '')) or
    ((CurPage = wpSelectDir) and ((shDisableDirPage in SetupHeader.Options) or not(shCreateAppDir in SetupHeader.Options))) or
    ((CurPage = wpSelectComponents) and not HasComponents) or
    ((CurPage = wpSelectProgramGroup) and ((shDisableProgramGroupPage in SetupHeader.Options) or not HasTasklessIcons)) or
    ((CurPage = wpSelectTasks) and SkipSelectTasksPage()) or
    ((CurPage = wpReady) and (shDisableReadyPage in SetupHeader.Options));

  if (Result = False) and not (CurPage in [wpWelcome]) then
    if HasCodeText then
      Result := ScriptRunBooleanFunction('SkipCurPage', ['CurPage', Integer(CurPage)], False, Result);
end;

procedure TWizardForm.NextButtonClick(Sender: TObject);

  function CheckPassword: Boolean;
  var
    S: String;
    SaveCursor: HCURSOR;
  begin
    S := PasswordEdit.Text;
    Result := GetCRC32(S[1], Length(S)) = SetupHeader.Password;
    if Result then
      NeedPassword := False
    else begin
      { Delay for 750 ms when an incorrect password is entered to
        discourage brute-force attempts }
      SaveCursor := GetCursor;
      SetCursor (LoadCursor(0, IDC_WAIT));
      Sleep (750);
      SetCursor (SaveCursor);
      MsgBox (SetupMessages[msgIncorrectPassword], '', mbError, MB_OK);
      PasswordEdit.Text := '';
      PasswordEdit.SetFocus;
    end;
  end;

  function GetTotalFreeSpace (const Drive: Char; var TotalSpace: Longint): Boolean;
  var
    SectorsPerCluster, BytesPerSector, FreeClusters, TotalClusters, Temp: Cardinal;
  begin
    Result := GetDiskFreeSpace(PChar(Drive + ':\'), DWORD(SectorsPerCluster),
      DWORD(BytesPerSector), DWORD(FreeClusters), DWORD(TotalClusters));
    if Result then begin
      Temp := BytesPerSector * SectorsPerCluster;
      { Windows 95/98 cap the result of GetDiskFreeSpace at 2GB, but NT 4.0
        does not, so we must use a 64-bit multiply operation to avoid an
        overflow. Delphi versions < 4 don't natively support 64-bit multiplies,
        so use assembler code to do this. }
      asm
        mov eax, Temp
        mul FreeClusters     { Multiplies EAX by FreeClusters, places 64-bit result in EDX:EAX }
        test edx, edx        { EDX will be zero if result is < 4 GB }
        jz @@1
        mov eax, 0FFFFFFFFh  { If > 4 GB then cap at 4 GB }
        @@1:
        mov Temp, eax
      end;
      TotalSpace := Temp;
    end;
  end;

  function SpaceString (const S: String): String;
  var
    I: Integer;
  begin
    Result := '';
    for I := 1 to Length(S) do begin
      if S[I] = ' ' then Continue;
      if Result <> '' then Result := Result + ' ';
      Result := Result + S[I];
    end;
  end;

  function CheckSelectDirPage: Boolean;
  const
    BadDirChars3 = '/:*?"<>|,()[]';
    BadDirChars4 = '/:*?"<>|';
  var
    T: String;
    I: Integer;
    TotalFreeSpace: Longint;
    BadDirChars: String;
  begin
    Result := False;

    { Remove any leading or trailing spaces }
    DirEdit.Text := Trim(DirEdit.Text);

    T := DirEdit.Text;  { reduce calls to GetText }

    { Check for UNC pathname }
    if Pos('\\', T) <> 0 then begin
      MsgBox (SetupMessages[msgToUNCPathname], '', mbError, MB_OK);
      Exit;
    end;

    { Check if is at least 4 chars long and it has a colon and backslash }
    if not(shAllowRootDirectory in SetupHeader.Options) then
      I := 4
    else
      I := 3;
    if (Length(T) < I) or (T[2] <> ':') or (T[3] <> '\') then begin
      MsgBox (SetupMessages[msgInvalidPath], '', mbError, MB_OK);
      Exit;
    end;

    { Check for invalid characters after x:\ }
    if UsingWindows4 then
      BadDirChars := BadDirChars4
    else
      BadDirChars := BadDirChars3;
    for I := 4 to Length(T) do
      if Pos(T[I], BadDirChars) <> 0 then begin
        MsgBox (FmtSetupMessage1(msgBadDirName32, SpaceString(BadDirChars)), '',
          mbError, MB_OK);
        Exit;
      end;

    { Check if it's a valid drive }
    if not(UpCase(T[1]) in ['A'..'Z']) or not GetTotalFreeSpace(T[1], TotalFreeSpace) then begin
      MsgBox (SetupMessages[msgInvalidDrive], '', mbError, MB_OK);
      Exit;
    end;

    { Remove trailing backslash }
    T := RemoveBackslashUnlessRoot(T);
    DirEdit.Text := T;

    { Check if enough disk space (using an unsigned long compare) }
    if Cardinal(TotalFreeSpace) < Cardinal(MinimumSpace) then
      { If not, show warning }
      if MsgBox(FmtSetupMessage(msgDiskSpaceWarning,
           [IntToStr(MinimumSpace div 1024), IntToStr(TotalFreeSpace div 1024)]),
         SetupMessages[msgDiskSpaceWarningTitle],
         mbConfirmation, MB_YESNO or MB_DEFBUTTON2) <> ID_YES then
        Exit;

    { Check if directory already exists }
    if ((SetupHeader.DirExistsWarning = ddYes) or
        ((SetupHeader.DirExistsWarning = ddAuto) and (T <> PreviousAppDir))) and
       DirExists(T) then
      { If so, ask if user wants to install there anyway }
      if MsgBox(FmtSetupMessage1(msgDirExists, T), SetupMessages[msgDirExistsTitle],
        mbConfirmation, MB_YESNO) <> ID_YES then Exit;

    { Check if directory *doesn't* already exist }
    if (shEnableDirDoesntExistWarning in SetupHeader.Options) and
       not DirExists(T) then
      { If not, ask if user wants to install there anyway }
      if MsgBox(FmtSetupMessage1(msgDirDoesntExist, T), SetupMessages[msgDirDoesntExistTitle],
        mbConfirmation, MB_YESNO) <> ID_YES then Exit;

    Result := True;
  end;

  function CheckSelectComponentsPage: Boolean;
  var
    ComponentEntry: PSetupComponentEntry;
    TotalFreeSpace: Longint;
    S: String;
    I: Integer;
  begin
    Result := False;

    if GetTotalFreeSpace(DirEdit.Text[1], TotalFreeSpace) then begin
      if Cardinal(TotalFreeSpace) < Cardinal(MinimumComponentsSpace) then
        if MsgBox(FmtSetupMessage(msgDiskSpaceWarning,
             [IntToStr(MinimumComponentsSpace div 1024), IntToStr(TotalFreeSpace div 1024)]),
           SetupMessages[msgDiskSpaceWarningTitle],
           mbConfirmation, MB_YESNO or MB_DEFBUTTON2) <> ID_YES then
          Exit;
    end;

    //now see if there are unchecked components that are already installed
    if PrevSelectedComponents.Count > 0 then begin
      S := '';
      for I := 0 to ComponentsList.Items.Count-1 do begin
        if not ComponentsList.Checked[I] then begin
          ComponentEntry := PSetupComponentEntry(ComponentsList.ItemObject[I]);
          if not (coDisableNoUninstallWarning in ComponentEntry.Options) then begin
            if PrevSelectedComponents.IndexOf(ComponentEntry.Name) <> -1 then begin
              if S <> '' then
                S := S + #13;
              S := S + ComponentEntry.Description;
            end;
          end;
        end;
      end;

      if (S = '') or (MsgBox(FmtSetupMessage1(msgNoUninstallWarning, S), SetupMessages[msgNoUninstallWarningTitle], mbConfirmation, MB_YESNO) = ID_YES) then
        Result := True;
    end else
      Result := True;
  end;

  function CheckSelectProgramGroupPage: Boolean;
  const
    BadGroupCharsOld = '\/:*?"<>|,';
    BadGroupCharsNew = '/:*?"<>|,';
  var
    T, BadGroupChars: String;
    I: Integer;
  begin
    Result := False;

    if not NoIconsCheck.Checked then begin
      { Remove any leading or trailing spaces and out of place backslashes }
      T := Trim(GroupEdit.Text);
      I := Pos('\\', T);
      while I <> 0 do begin
        Delete (T, I, 1);
        I := Pos('\\', T);
      end;
      if (T <> '') and (T[1] = '\') then
        Delete (T, 1, 1);
      T := RemoveBackslash(T);
      GroupEdit.Text := T;

      if T = '' then begin
        MsgBox (SetupMessages[msgMustEnterGroupName], '', mbError, MB_OK);
        Exit;
      end;

      { Check for invalid characters. Quotes are not allowed because the
        PROGMAN DDE interface uses them to delimit command strings. And
        since Windows 95/NT 4.0 stores program groups as literal directory
        names, quotes and the rest of the characters in the BadGroupChars
        constant aren't allowed either. }
      if not UsingWindows4 then
        BadGroupChars := BadGroupCharsOld
      else
        BadGroupChars := BadGroupCharsNew;
      for I := 1 to Length(T) do
        if Pos(T[I], BadGroupChars) <> 0 then begin
          MsgBox (FmtSetupMessage1(msgBadGroupName, SpaceString(BadGroupChars)),
            '', mbError, MB_OK);
          Exit;
        end;
    end;

    Result := True;
  end;

  procedure UpdateReadyPage;
  var
    TypeEntry: PSetupTypeEntry;
    SelectedComponents, SelectedTasks: TStringList;
    Space: String;
    I: Integer;
  begin
    if ReadyMemo.Visible then begin
      Space := Format('%6s', ['']);
      ReadyMemo.Lines.BeginUpdate();
      ReadyMemo.Lines.Clear();

      ReadyMemo.Lines.Append(SetupMessages[msgReadyMemoDir]);
      ReadyMemo.Lines.Append(Space+DirEdit.Text);

      if HasComponents then begin
        ReadyMemo.Lines.Append('');
        ReadyMemo.Lines.Append(SetupMessages[msgReadyMemoType]);
        TypeEntry := PSetupTypeEntry(TypesCombo.Items.Objects[TypesCombo.ItemIndex]);
        ReadyMemo.Lines.Append(Space+TypeEntry.Description);

        SelectedComponents := TStringList.Create();
        GetSelectedComponents(SelectedComponents, True);
        if SelectedComponents.Count > 0 then begin
          ReadyMemo.Lines.Append('');
          ReadyMemo.Lines.Append(SetupMessages[msgReadyMemoComponents]);
          for I := 0 to SelectedComponents.Count-1 do
            ReadyMemo.Lines.Append(Space+SelectedComponents[I]);
        end;
        SelectedComponents.Free();
      end;

      if HasTasklessIcons and not NoIconsCheck.Checked then begin
        ReadyMemo.Lines.Append('');
        ReadyMemo.Lines.Append(SetupMessages[msgReadyMemoGroup]);
        ReadyMemo.Lines.Append(Space+GroupEdit.Text);
      end;

      SelectedTasks := TStringList.Create();
      GetSelectedTasks(SelectedTasks, True);
      if SelectedTasks.Count > 0 then begin
        ReadyMemo.Lines.Append('');
        ReadyMemo.Lines.Append(SetupMessages[msgReadyMemoTasks]);
        for I := 0 to SelectedTasks.Count-1 do
          ReadyMemo.Lines.Append(Space+SelectedTasks[I]);
      end;
      SelectedTasks.Free();

      ReadyMemo.SelStart := 0;
      ReadyMemo.SelLength := 0;

      ReadyMemo.Lines.EndUpdate();
    end;
  end;

begin
  case CurPage of
    wpPassword: if not CheckPassword then Exit;
    wpSelectDir: if not CheckSelectDirPage then Exit;
    wpSelectComponents: if not CheckSelectComponentsPage then Exit;
    wpSelectProgramGroup: if not CheckSelectProgramGroupPage then Exit;
  end;

  if HasCodeText then
    if ScriptRunBooleanFunction('NextButtonClick', ['CurPage', Integer(CurPage)], False, True) = False then
      Exit;

  { Go to the next page, or close wizard if it was on the last page }
  repeat
    {$IFDEF WIZARDCLASSIC}
    if CurPage in [wpReady, wpFinished] then begin
    {$ELSE}
    if CurPage = wpReady then begin
      CurPage := wpInstalling;
      CurPageChanged;
      PostMessage (MainForm.Handle, WM_SETUPNEXTSTEP, 0, 0);
      Exit;
    end else if CurPage = wpFinished then begin
    {$ENDIF}
      CantShow := True;
      DisableCloseQuery := True;
      Close;
      DisableCloseQuery := False;
      PostMessage (MainForm.Handle, WM_SETUPNEXTSTEP, 0, 0);
      Exit;
    end;

    Inc (CurPage);
    //even if we're skipping a page, we should still update it
    case CurPage of
      wpSelectTasks: begin
          if NeedSelectTasksPageUpdate then begin
            UpdateSelectTasksPage();
            NeedSelectTasksPageUpdate := False;
          end;
        end;
      wpReady: UpdateReadyPage;
    end;
  until not SkipCurPage;

  CurPageChanged;
end;

procedure TWizardForm.BackButtonClick(Sender: TObject);
begin
  if CurPage = Low(TWizardPage) then Exit;

  if HasCodeText then
    if ScriptRunBooleanFunction('BackButtonClick', ['CurPage', Integer(CurPage)], False, True) = False then
      Exit;

  { Go to the previous page }
  Dec (CurPage);
  while SkipCurPage do
    Dec (CurPage);
  CurPageChanged;
end;

procedure TWizardForm.CancelButtonClick(Sender: TObject);
begin
  { Clicking Cancel will do the same thing as the Close button }
  Close;
end;

procedure TWizardForm.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
  if not DisableCloseQuery then begin
    DisableCloseQuery := True;
    try
      { Redirect an attempt to close this form to MainForm }
      MainForm.Close;
      { And only close if MainForm closed }
      CanClose := MainForm.Closing;
    finally
      DisableCloseQuery := False;
    end;
  end;
end;

procedure TWizardForm.ReadGroupList;
{ Fills in the list box of existing groups }
var
  PROGMAN, Item: TDDEString;
  Conv: TDDEConversation;
  DataHandle: TDDEData;
  DataPointer: PChar;
  DataSize: LongInt;
  H: THandle;
  FindData: TWin32FindData;
  S: String;
begin
  try
    if not UsingWindows4 then begin
      PROGMAN := DDE.CreateString('PROGMAN');
      Item := DDE.CreateString('GROUPS');
      Conv := DDE.BeginConnection(PROGMAN, PROGMAN);
      try
        DDE.RequestBegin (Conv, Item, CF_TEXT, DataHandle, Pointer(DataPointer), DataSize);
        try
          GroupList.Items.SetText (DataPointer);
        finally
          DDE.RequestEnd (DataHandle);
        end;
      finally
        DDE.EndConnection (Conv);
        DDE.FreeString (Item);
        DDE.FreeString (PROGMAN);
      end;
    end
    else begin
      H := FindFirstFile(PChar(ChangeDirConst('{group}\*')), FindData);
      if H <> INVALID_HANDLE_VALUE then begin
        repeat
          if FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY <> 0 then begin
            S := FindData.cFileName;
            if (S <> '.') and (S <> '..') then
              GroupList.Items.Add (S);
          end;
        until not FindNextFile(H, FindData);
        WinProcs.FindClose (H);
      end;
    end;
  except
    { ignore any exceptions }
  end;
end;

procedure TWizardForm.DirListChange(Sender: TObject);
begin
  if not(shDisableAppendDir in SetupHeader.Options) then
    DirEdit.Text := AddBackslash(DirList.Directory) +
      ExtractFileName(ChangeDirConst(SetupHeader.DefaultDirName))
  else
    DirEdit.Text := DirList.Directory;
end;

procedure TWizardForm.TypesComboChange(Sender: TObject);
var
  TypeEntry: PSetupTypeEntry;
begin
  //select the components for this type. if the type is custom only select
  //fixed components
  TypeEntry := PSetupTypeEntry(TypesCombo.Items.Objects[TypesCombo.ItemIndex]);
  SetSelectedComponentsFromType(TypeEntry.Name, (toIsCustom in TypeEntry.Options));

  //if customization is possible remember the type and components that are
  //selected, so that we can reselect the setup type later if after customization
  //the user didn't really change anything
  //also hide the components list if necessary
  if HasCustomType then begin
    InitialSetupTypeIndex := TypesCombo.ItemIndex;
    GetSelectedComponents(InitialSelectedComponents, False);
    if not (shAlwaysShowComponentsList in SetupHeader.Options) then begin
      ComponentsList.Visible := toIsCustom in TypeEntry.Options;
      ComponentsDiskSpaceLabel.Visible := ComponentsList.Visible;
    end;
  end;

  CalcComponentsSpace();
  NeedSelectTasksPageUpdate := True;
end;

procedure TWizardForm.ComponentsListClickCheck(Sender: TObject);
var
  SelectedComponents: TStringList;
  TypeEntry: PSetupTypeEntry;
  Equals: Boolean;
  I: Integer;
begin
  //first see if this current selection equals the initial selection
  //if so, reselect the initial setup type
  SelectedComponents := TStringList.Create();
  GetSelectedComponents(SelectedComponents, False);
  Equals := SelectedComponents.Equals(InitialSelectedComponents);
  SelectedComponents.Free();

  if Equals then begin
    //select the intial type
    TypesCombo.ItemIndex := InitialSetupTypeIndex;
  end else begin
    //select a custom type
    for I := 0 to Entries[seType].Count-1 do begin
      TypeEntry := Entries[seType][I];
      if (toIsCustom in TypeEntry.Options) then begin
        TypesCombo.ItemIndex := TypesCombo.Items.IndexOfObject(TObject(TypeEntry));
        SetSelectedComponentsFromType(TypeEntry.Name, True);
        Break;
      end;
    end
  end;

  CalcComponentsSpace();
  NeedSelectTasksPageUpdate := True;
end;

procedure TWizardForm.GroupEditChange(Sender: TObject);
var
  T: String;
  I: Integer;
  SaveOnClick: TNotifyEvent;
label 1;
begin
  T := GroupEdit.Text;
  with GroupList do begin
    for I := 0 to Items.Count-1 do
      if AnsiCompareText(T, Items[I]) = 0 then
        goto 1;
    I := -1;
  1:if ItemIndex <> I then begin
      SaveOnClick := OnClick;
      { Temporarily disabled the OnChange event }
      OnClick := nil;
      try
        ItemIndex := I;
      finally
        OnClick := SaveOnClick;
      end;
    end;
  end;
end;

procedure TWizardForm.GroupListClick(Sender: TObject);
var
  SaveOnChange: TNotifyEvent;
begin
  with GroupList do
    if ItemIndex <> -1 then begin
      SaveOnChange := GroupEdit.OnChange;
      { Temporarily disabled the OnChange event }
      GroupEdit.OnChange := nil;
      try
        GroupEdit.Text := Items[ItemIndex];
      finally
        GroupEdit.OnChange := SaveOnChange;
      end;
    end;
end;

procedure TWizardForm.NoIconsCheckClick(Sender: TObject);
const
  ColorChange: array[Boolean] of TColor = (clBtnFace, clWindow);
begin
  with GroupEdit do begin
    Enabled := not NoIconsCheck.Checked;
    Color := ColorChange[Enabled];
  end;
  with GroupList do begin
    Enabled := not NoIconsCheck.Checked;
    Color := ColorChange[Enabled];
    if not Enabled then ItemIndex := -1;
  end;
end;

procedure TWizardForm.ReadyMemoKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then
    NextButton.Click();
end;

{$IFDEF WIZARDCLASSIC}

procedure TWizardForm.PicturePaintBoxPaint(Sender: TObject);
var
  R: TRect;
  X, Y: Integer;
begin
  with PicturePaintBox do begin
    R := ClientRect;
    { Draw the "sunken" edge, and exclude the edge from the clipping region }
    NewDrawEdge (Canvas, R, deThinSunken);
    InflateRect (R, -1, -1);
    IntersectClipRect (Canvas.Handle, R.Left, R.Top, R.Right, R.Bottom);
    { Draw the teal background }
    Canvas.Brush.Color := SetupHeader.WizardImageBackColor;
    Canvas.FillRect (R);
    { And finally draw the image }
    if Assigned(WizardImage) then begin
      X := R.Left + ((R.Right - R.Left) - WizardImage.Width) div 2;
      if X < 0 then X := 0;
      Y := R.Top + ((R.Bottom - R.Top) - WizardImage.Height) div 2;
      if Y < 0 then Y := 0;
      Canvas.Draw (X, Y, WizardImage);
    end;
  end;
end;

procedure TWizardForm.SetupIconPaint(Sender: TObject);
begin
  (Sender as TPaintBox).Canvas.Draw (0, 0, Application.Icon);
end;

{$ELSE}

procedure TWizardForm.PicturePaintBoxPaint(Sender: TObject);
var
  Bitmap: TBitmap;
  BackColor: TColor;
  R: TRect;
  X, Y: Integer;
begin
  if Sender = SmallPicturePaintBox then begin
    Bitmap := WizardSmallImage;
    BackColor := SetupHeader.WizardSmallImageBackColor;
  end else begin
    Bitmap := WizardImage;
    BackColor := SetupHeader.WizardImageBackColor;
  end;

  with TPaintBox(Sender) do begin
    R := ClientRect;
    if Sender <> SmallPicturePaintBox then begin
      Canvas.Pen.Color := clBtnFace;
      Canvas.MoveTo(R.Right-1, 0);
      Canvas.LineTo(R.Right-1, R.Bottom);
      Dec(R.Right);
    end;

    IntersectClipRect (Canvas.Handle, R.Left, R.Top, R.Right, R.Bottom);
    { Draw the teal background }
    Canvas.Brush.Color := BackColor;
    Canvas.FillRect (R);
    { And finally draw the image }
    if Assigned(Bitmap) then begin
      X := R.Left + ((R.Right - R.Left) - Bitmap.Width) div 2;
      if X < 0 then X := 0;
      Y := R.Top + ((R.Bottom - R.Top) - Bitmap.Height) div 2;
      if Y < 0 then Y := 0;
      Canvas.Draw (X, Y, Bitmap);
    end;
  end;
end;

{$ENDIF}

procedure TWizardForm.WMSysCommand (var Message: TWMSysCommand);
begin
  if Message.CmdType and $FFF0 = SC_MINIMIZE then
    { A minimize button is shown on the wizard form when (shWindowVisible in
      SetupHeader.Options). When it is clicked we want to minimize the whole
      application. }
    Application.Minimize
  else
  if Message.CmdType = 9999 then
    MainForm.ShowAboutBox
  else
    inherited;
end;
