unit Main;

{
  Inno Setup
  Copyright (C) 1998-2000 Jordan Russell
  For conditions of distribution and use, see LICENSE.TXT.

  Background form
}

interface

{$I VERSION.INC}

uses
  WinTypes, WinProcs, SysUtils, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls,
  Struct;

const
  WM_SETUPNEXTSTEP = WM_USER + 3900;

type
  TCurStep = (csStart, csWizard, csCopy, csFinished, csTerminate);
  TMainForm = class(TForm)
    procedure FormResize(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormActivate(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
  private
    { Private declarations }
    procedure AppOnActivate (Sender: TObject);
    procedure AppOnRestore (Sender: TObject);
    function MainWindowHook (var Message: TMessage): Boolean;
    procedure WMSysCommand (var Message: TWMSysCommand); message WM_SYSCOMMAND;
    procedure WMSetupNextStep (var Msg: TMessage); message WM_SETUPNEXTSTEP;
    procedure WMEraseBkgnd (var Message: TWMEraseBkgnd); message WM_ERASEBKGND;
    procedure WMGetDlgCode (var Message: TWMGetDlgCode); message WM_GETDLGCODE;
  protected
    function GetPalette: HPALETTE; override;
  public
    { Public declarations }
    CurStep: TCurStep;
    Closing, IsMinimized, InExecuteLoop: Boolean;
    NewPalette: HPALETTE;
    constructor Create (AOwner: TComponent); override;
    destructor Destroy; override;
    procedure CenterForm (AForm: TForm);
    procedure ShowAboutBox;
  end;

  TInstallOnThisVersionResult = (irInstall, irNotOnThisPlatform,
    irVerTooLow, irVerTooHigh);
  TShellFolderID = (sfDesktop, sfStartMenu, sfPrograms, sfStartup, sfSendTo,
    sfFonts, sfAppData, sfDocs, sfTemplates, sfFavorites, sfLocalAppData);
  TEntryType = (seType, seComponent, seTask, seDir, seFile, seFileLocation, seIcon, seIni, seRegistry,
    seInstallDelete, seUninstallDelete, seRun, seUninstallRun);

const
  EntryStrings: array[TEntryType] of Integer = (SetupTypeEntryStrings, SetupComponentEntryStrings, SetupTaskEntryStrings, SetupDirEntryStrings,
    SetupFileEntryStrings, SetupFileLocationEntryStrings, SetupIconEntryStrings,
    SetupIniEntryStrings, SetupRegistryEntryStrings, SetupDeleteEntryStrings,
    SetupDeleteEntryStrings, SetupRunEntryStrings, SetupRunEntryStrings);

var
  MainForm: TMainForm;

  { Variables for command line parameters }
  SetupLdrMode: Boolean;
  SetupLdrWnd: HWND;
  SetupLdrOriginalFilename: String;
  SetupLdrOffsetMsg, SetupLdrOffset0, SetupLdrOffset1: Longint;
  InitDir, InitProgramGroup: String;
  InitLoadInf, InitSaveInf: String;
  InitNoIcons, InitSilent, InitVerySilent, InitNoRestart: Boolean;
  InitComponents: TStringList;
  DetachedUninstMsgFile: Boolean;

  { 'Constants' }
  SourceDir, TempInstallDir, WinDir, WinSystemDir, SystemDrive, FontDir,
    ProgramFilesDir, CommonFilesDir, DAODir: String;

  { Variables read in from the SETUP.0 file }
  SetupHeader: TSetupHeader;
  Entries: array[TEntryType] of TList;
  WizardImage: TBitmap;
  WizardSmallImage: TBitmap;

  { Other }
  HasIcons, IsNT, NewGUI, IsAdmin, NeedPassword, NeedsRestart,
    RestartSystem: Boolean;
  HasCustomType, HasComponents, HasTasks, HasTaskIcons,
    HasTasklessIcons, HasCodeText: Boolean;
  RunCheckBoxes: TList;
  WindowsVersion: Cardinal;
  NTServicePackLevel: Word;
  MinimumSpace: Longint;
  DeleteFilesAfterInstallList, DeleteDirsAfterInstallList: TStringList;
  UninstallRegKeyBaseName: String;
  StartParam: Integer;

function ShouldProcessEntry(const WizardComponents, WizardTasks: TStringList; const Components, Tasks, Check: String): Boolean;
function ShouldProcessRunEntry(const WizardComponents, WizardTasks: TStringList; const RunEntry: PSetupRunEntry): Boolean;
function ShouldProcessIconEntry(const WizardComponents, WizardTasks: TStringList; const WizardNoIcons: Boolean; const IconEntry: PSetupIconEntry): Boolean;
function ChangeDirConst (const S: String): String;
function ChangeDirConstEx (const S: String; const CustomConsts: array of String): String;
procedure DeinitSetup;
function ExitSetupMsgBox: Boolean;
function GetShellFolder (Common: Boolean; const ID: TShellFolderID): String;
procedure InitializeSetup;
function InstallOnThisVersion (const MinVersion: TSetupVersionData;
  const OnlyBelowVersion: TSetupVersionData): TInstallOnThisVersionResult;
procedure SetFontNameSize (const AFont: TFont; const ANameSize: String;
  const ADefaultName: String; const ADefaultSize: Integer);
procedure InternalError (const Id: String);
procedure SetFormFont (Form: TForm; var OldTextHeight, NewTextHeight: Integer);

implementation

uses
  ShellAPI, ShlObj,
  Msgs, MsgIDs, ChildFrm, Install, InstFunc, InstFnc2, CmnFunc, CmnFunc2, Script,
  zlib, SetupEnt, {$IFDEF WIZARDCLASSIC} WizardClassic; {$ELSE} WizardModern; {$ENDIF}


{$R *.DFM}

var
  ShellFolders: array[Boolean, TShellFolderID] of String;
  ShellFoldersRead: array[Boolean, TShellFolderID] of Boolean;
  SHFolderDLLHandle, SHFolderShell32Handle: HMODULE;
  SHGetFolderPathFunc: function(hwndOwner: HWND; nFolder: Integer;
    hToken: THandle; dwFlags: DWORD; pszPath: PAnsiChar): HRESULT; stdcall;

const
  CSIDL_FLAG_CREATE = $8000;
  SHGFP_TYPE_CURRENT = 0;
  SHGFP_TYPE_DEFAULT = 1;

{ Misc. functions }

function ShouldProcessEntry(const WizardComponents, WizardTasks: TStringList; const Components, Tasks, Check: String): Boolean;

  function ShouldProcessEntry(const WizardItems, EntryItems: TStringList): Boolean;
  var
    I: Integer;
  begin
    Result := False;
    for I := 0 to WizardItems.Count-1 do begin
      if EntryItems.IndexOf(WizardItems[I]) <> -1 then begin
        Result := True;
        Break;
      end;
    end;
  end;

var
  ProcessComponent, ProcessTask: Boolean;
  EntryItems: TStringList;
begin
  if (Components <> '') or (Tasks <> '') or (Check <> '') then begin
    EntryItems := TStringList.Create();

    if (Components <> '') and (WizardComponents <> nil) then begin
      EntryItems.CommaText := Components;
      ProcessComponent := ShouldProcessEntry(WizardComponents, EntryItems);
    end else
      ProcessComponent := True;

    if (Tasks <> '') and (WizardTasks <> nil) then begin
      EntryItems.CommaText := Tasks;
      ProcessTask := ShouldProcessEntry(WizardTasks, EntryItems);
    end else
      ProcessTask := True;

    EntryItems.Free();

    Result := ProcessComponent and ProcessTask;
    if (Result = True) and (Check <> '') then
      Result := ScriptRunBooleanFunction(Check, [''], True, Result);
  end else
    Result := True;
end;

function ShouldProcessRunEntry(const WizardComponents, WizardTasks: TStringList; const RunEntry: PSetupRunEntry): Boolean;
begin
  if (SetupHeader.InstallMode <> imNormal) and (roSkipIfSilent in RunEntry.Options) then
    Result := False
  else if (SetupHeader.InstallMode = imNormal) and (roSkipIfNotSilent in RunEntry.Options) then
    Result := False
  else
    Result := ShouldProcessEntry(WizardComponents, WizardTasks, RunEntry.Components, RunEntry.Tasks, RunEntry.Check);
end;

function ShouldProcessIconEntry(const WizardComponents, WizardTasks: TStringList; const WizardNoIcons: Boolean; const IconEntry: PSetupIconEntry): Boolean;
begin
  if WizardNoIcons and (IconEntry.Tasks = '') then
    Result := False
  else
    Result := ShouldProcessEntry(WizardComponents, WizardTasks, IconEntry.Components, IconEntry.Tasks, IconEntry.Check);
end;

procedure LoadInf(const FileName: String);
var
  Section: String;
begin
  Section := 'Setup';
  InitDir := GetIniString(Section, 'Dir', InitDir, FileName);
  InitProgramGroup := GetIniString(Section, 'Group', InitProgramGroup, FileName);
  InitNoIcons := GetIniBool(Section, 'NoIcons', InitNoIcons, FileName);
  InitSilent := GetIniBool(Section, 'Silent', InitSilent, FileName);
  InitVerySilent := GetIniBool(Section, 'VerySilent', InitVerySilent, FileName);
  InitNoRestart := GetIniBool(Section, 'NoRestart', InitNoRestart, FileName);
  InitComponents.CommaText := GetIniString(Section, 'Components', InitComponents.CommaText, FileName);
  InitSaveInf := GetIniString(Section, 'SaveInf', InitSaveInf, FileName);
end;

procedure SaveInf(const FileName: String);
var
  Section: String;
begin
  Section := 'Setup';
  SetIniString(Section, 'Dir', WizardDirValue, FileName);
  SetIniString(Section, 'Group', WizardGroupValue, FileName);
  SetIniBool(Section, 'NoIcons', WizardNoIcons, FileName);
  SetIniString(Section, 'Components', WizardComponents.CommaText, FileName);
end;

function ChangeIndividualConst (const Cnst: String; const CustomConsts: array of String;
  var IsPath: Boolean): String;
{ Cnst must be the name of a single constant, without the braces.
  For example: app
  IsPath is set to True if the result is a path which needs special trailing-
  backslash handling. }

  function ExpandRegConst (C: String): String;
  type
    TKeyNameConst = packed record
      KeyName: String;
      KeyConst: HKEY;
    end;
  const
    KeyNameConsts: array[0..4] of TKeyNameConst = (
      (KeyName: 'HKCR'; KeyConst: HKEY_CLASSES_ROOT),
      (KeyName: 'HKCU'; KeyConst: HKEY_CURRENT_USER),
      (KeyName: 'HKLM'; KeyConst: HKEY_LOCAL_MACHINE),
      (KeyName: 'HKU';  KeyConst: HKEY_USERS),
      (KeyName: 'HKCC'; KeyConst: HKEY_CURRENT_CONFIG));
  var
    Z, Subkey, Value, Default: String;
    I, J: Integer;
    RootKey: HKEY;
    K: HKEY;
  begin
    Delete (C, 1, 4);  { skip past 'reg:' }
    I := ConstPos('\', C);
    if I <> 0 then begin
      Z := Copy(C, 1, I-1);
      if Z <> '' then begin
        RootKey := 0;
        for J := Low(KeyNameConsts) to High(KeyNameConsts) do
          if CompareText(KeyNameConsts[J].KeyName, Z) = 0 then begin
            RootKey := KeyNameConsts[J].KeyConst;
            Break;
          end;
        if RootKey <> 0 then begin
          Z := Copy(C, I+1, Maxint);
          I := ConstPos('|', Z);  { check for a 'default' data }
          if I = 0 then
            I := Length(Z)+1;
          Default := Copy(Z, I+1, Maxint);
          SetLength (Z, I-1);
          I := ConstPos(',', Z);  { comma separates subkey and value }
          if I <> 0 then begin
            Subkey := Copy(Z, 1, I-1);
            Value := Copy(Z, I+1, Maxint);
            if ConvertConstPercentStr(Subkey) and ConvertConstPercentStr(Value) and
               ConvertConstPercentStr(Default) then begin
              Result := ChangeDirConstEx(Default, CustomConsts);
              if RegOpenKeyEx(RootKey, PChar(ChangeDirConstEx(Subkey, CustomConsts)),
                 0, KEY_QUERY_VALUE, K) = ERROR_SUCCESS then begin
                RegQueryStringValue (K, PChar(ChangeDirConstEx(Value, CustomConsts)),
                  Result);
                RegCloseKey (K);
              end;
              Exit;
            end;
          end;
        end;
      end;
    end;
    { it will only reach here if there was a parsing error }
    InternalError ('C103');
  end;

  function ExpandIniConst (C: String): String;
  var
    Z, Filename, Section, Key, Default: String;
    I: Integer;
  begin
    Delete (C, 1, 4);  { skip past 'ini:' }
    I := ConstPos(',', C);
    if I <> 0 then begin
      Z := Copy(C, 1, I-1);
      if Z <> '' then begin
        Filename := Z;
        Z := Copy(C, I+1, Maxint);
        I := ConstPos('|', Z);  { check for a 'default' data }
        if I = 0 then
          I := Length(Z)+1;
        Default := Copy(Z, I+1, Maxint);
        SetLength (Z, I-1);
        I := ConstPos(',', Z);  { comma separates section and key }
        if I <> 0 then begin
          Section := Copy(Z, 1, I-1);
          Key := Copy(Z, I+1, Maxint);
          if ConvertConstPercentStr(Filename) and ConvertConstPercentStr(Section) and ConvertConstPercentStr(Key) and
             ConvertConstPercentStr(Default) then begin
            Filename := ChangeDirConstEx(Filename, CustomConsts);
            Section := ChangeDirConstEx(Section, CustomConsts);
            Key := ChangeDirConstEx(Key, CustomConsts);
            Default := ChangeDirConstEx(Default, CustomConsts);
            Result := GetIniString(Section, Key, Default, Filename);
            Exit;
          end;
        end;
      end;
    end;
    { it will only reach here if there was a parsing error }
    InternalError ('C103');
  end;

  function ExpandParamConst (C: String): String;

    function GetParamString(const StartParam: Integer; const Param, Default: String): String;
    var
      I, PCount: Integer;
      Z: String;
    begin
      PCount := NewParamCount();
      for i := StartParam to PCount do begin
        Z := NewParamStr(i);
        if StrLIComp(PChar(Z), PChar('/'+Param+'='), Length(Param)+2) = 0 then begin
          Delete(Z, 1, Length(Param)+2);
          Result := Z;
          Exit;
        end;
      end;

      Result := Default;
    end;

  var
    Z, Param, Default: String;
    I: Integer;
  begin
    Delete (C, 1, 6);  { skip past 'param:' }
    Z := C;
    I := ConstPos('|', Z);  { check for a 'default' data }
    if I = 0 then
      I := Length(Z)+1;
    Default := Copy(Z, I+1, Maxint);
    SetLength (Z, I-1);
    Param := Z;
    if ConvertConstPercentStr(Param) and ConvertConstPercentStr(Default) then begin
      Param := ChangeDirConstEx(Param, CustomConsts);
      Default := ChangeDirConstEx(Default, CustomConsts);
      Result := GetParamString(StartParam, Param, Default);
      Exit;
    end;
    { it will only reach here if there was a parsing error }
    InternalError ('C103');
  end;

  function ExpandCodeConst (C: String): String;

    function GetCodeString(const ScriptFunc, Default: String): String;
    begin
      if HasCodeText then
        Result := ScriptRunStringFunction(ScriptFunc, ['Default', Default], True, Default)
      else
        Result := Default;
    end;

  var
    Z, Param, Default: String;
    I: Integer;
  begin
    Delete (C, 1, 5);  { skip past 'code:' }
    Z := C;
    I := ConstPos('|', Z);  { check for a 'default' data }
    if I = 0 then
      I := Length(Z)+1;
    Default := Copy(Z, I+1, Maxint);
    SetLength (Z, I-1);
    Param := Z;
    if ConvertConstPercentStr(Param) and ConvertConstPercentStr(Default) then begin
      Param := ChangeDirConstEx(Param, CustomConsts);
      Default := ChangeDirConstEx(Default, CustomConsts);
      Result := GetCodeString(Param, Default);
      Exit;
    end;
    { it will only reach here if there was a parsing error }
    InternalError ('C103');
  end;

const
  FolderConsts: array[Boolean, TShellFolderID] of String =
    (('userdesktop', 'userstartmenu', 'userprograms', 'userstartup',
      'sendto', 'fonts', 'userappdata', 'userdocs', 'usertemplates',
      'userfavorites', 'localappdata'),
     ('commondesktop', 'commonstartmenu', 'commonprograms', 'commonstartup',
      'sendto', 'fonts', 'commonappdata', 'commondocs', 'commontemplates',
      'commonfavorites', 'localappdata'));
var
  Z: String;
  B: Boolean;
  SF: TShellFolderID;
  K: Integer;
begin
  { Non-path constants }
  IsPath := False;
  if Cnst = '\' then Result := '\'
  else if Cnst = 'hwnd' then Result := IntToStr(MainForm.Handle)
  else if Cnst = 'computername' then Result := GetComputerNameString
  else if Cnst = 'username' then Result := GetUserNameString
  else if Cnst = 'groupname' then Result := WizardGroupValue
  else if Cnst = 'wizardhwnd' then begin
    if Assigned(WizardForm) then
      Result := IntToStr(WizardForm.Handle)
    else
      Result := '0';
  end else begin
    { Path constants }
    IsPath := True;
    if Cnst[1] = '%' then Result := GetEnv(Copy(Cnst, 2, Maxint))
    else if Copy(Cnst, 1, 4) = 'reg:' then Result := ExpandRegConst(Cnst)
    else if Copy(Cnst, 1, 4) = 'ini:' then Result := ExpandIniConst(Cnst)
    else if Copy(Cnst, 1, 6) = 'param:' then Result := ExpandParamConst(Cnst)
    else if Copy(Cnst, 1, 5) = 'code:' then Result := ExpandCodeConst(Cnst)
    else if Cnst = 'src' then Result := SourceDir
    else if Cnst = 'srcexe' then Result := SetupLdrOriginalFilename
    else if Cnst = 'tmp' then Result := TempInstallDir
    else if Cnst = 'app' then Result := WizardDirValue
    else if Cnst = 'win' then Result := WinDir
    else if Cnst = 'sys' then Result := WinSystemDir
    else if Cnst = 'sd' then Result := SystemDrive
    else if Cnst = 'pf' then Result := ProgramFilesDir
    else if Cnst = 'cf' then Result := CommonFilesDir
    else if Cnst = 'dao' then Result := DAODir
    else if Cnst = 'group' then begin
      Z := GetShellFolder(not(shAlwaysUsePersonalGroup in SetupHeader.Options), sfPrograms);
      if Z = '' then
        InternalError ('C100');
      Result := AddBackslash(Z) + WizardGroupValue;
    end
    else begin
      { Shell folder constants }
      for B := False to True do
        for SF := Low(SF) to High(SF) do
          if Cnst = FolderConsts[B, SF] then begin
            Z := GetShellFolder(B, SF);
            if Z = '' then begin
              if SF <> sfFonts then
                InternalError ('C101')
              else
                Result := FontDir;
            end
            else
              Result := Z;
            Exit;
          end;
        { Custom constants (non-path) }
        IsPath := False;
        if Cnst <> '' then begin
          K := 0;
          while K < High(CustomConsts) do begin
          if Cnst = CustomConsts[K] then begin
            Result := CustomConsts[K+1];
            Exit;
          end;
          Inc (K, 2);
        end;
      end;
      { Unknown constant }
      InternalError ('C102');
    end;
  end;
end;

function ChangeDirConst (const S: String): String;
begin
  Result := ChangeDirConstEx(S, ['']);
end;

function ChangeDirConstEx (const S: String; const CustomConsts: array of String): String;
var
  I, Start: Integer;
  Cnst, ReplaceWith: String;
  IsPath: Boolean;
begin
  Result := S;
  I := 1;
  while I <= Length(Result) do begin
    if Result[I] = '{' then begin
      if (I < Length(Result)) and (Result[I+1] = '{') then begin
        { Change '{{' to '{' if not in an embedded constant }
        Inc (I);
        Delete (Result, I, 1);
      end
      else begin
        Start := I;
        { Find the closing brace, skipping over any embedded constants }
        I := SkipPastConst(Result, I);
        if I = 0 then  { unclosed constant? }
          InternalError ('C104');
        Dec (I);  { 'I' now points to the closing brace }

        { Now translate the constant }
        Cnst := Copy(Result, Start+1, I-(Start+1));
        ReplaceWith := ChangeIndividualConst(Cnst, CustomConsts, IsPath);
        Delete (Result, Start, (I+1)-Start);
        Insert (ReplaceWith, Result, Start);
        I := Start + Length(ReplaceWith);
        if IsPath and (ReplaceWith <> '') and
           (ReplaceWith[Length(ReplaceWith)] = '\') and
           (I <= Length(Result)) and (Result[I] = '\') then
          Delete (Result, I, 1);
      end;
    end
    else
      Inc (I);
  end;
end;

function GetShellFolder (Common: Boolean; const ID: TShellFolderID): String;

  procedure GetFolder (const Common: Boolean);
  const
    CSIDL_COMMON_STARTMENU = $0016;
    CSIDL_COMMON_PROGRAMS = $0017;
    CSIDL_COMMON_STARTUP = $0018;
    CSIDL_COMMON_DESKTOPDIRECTORY = $0019;
    CSIDL_APPDATA = $001A;
    CSIDL_LOCAL_APPDATA = $001C;
    CSIDL_COMMON_FAVORITES = $001F;
    CSIDL_COMMON_APPDATA = $0023;
    CSIDL_COMMON_TEMPLATES = $002D;
    CSIDL_COMMON_DOCUMENTS = $002E;
    FolderIDs: array[Boolean, TShellFolderID] of Integer = (
      { User }
      (CSIDL_DESKTOPDIRECTORY, CSIDL_STARTMENU, CSIDL_PROGRAMS, CSIDL_STARTUP,
       CSIDL_SENDTO, CSIDL_FONTS, CSIDL_APPDATA, CSIDL_PERSONAL,
       CSIDL_TEMPLATES, CSIDL_FAVORITES, CSIDL_LOCAL_APPDATA),
      { Common }
      (CSIDL_COMMON_DESKTOPDIRECTORY, CSIDL_COMMON_STARTMENU, CSIDL_COMMON_PROGRAMS, CSIDL_COMMON_STARTUP,
       CSIDL_SENDTO, CSIDL_FONTS, CSIDL_COMMON_APPDATA, CSIDL_COMMON_DOCUMENTS,
       CSIDL_COMMON_TEMPLATES, CSIDL_COMMON_FAVORITES, CSIDL_LOCAL_APPDATA));
  var
    Z: String;
    Buf: array[0..MAX_PATH-1] of Char;
  begin
    if not ShellFoldersRead[Common, ID] then begin
      if ID in [sfDesktop, sfStartMenu, sfPrograms, sfStartup, sfSendTo,
         sfFonts, sfTemplates, sfFavorites] then
        { GetShellFolderPath calls SHGetSpecialFolderLocation }
        Z := GetShellFolderPath(FolderIDs[Common, ID])
      else begin
        { Only use SHGetFolderPath on CSIDLs unrecognized by Windows 95,
          the reason being that when CSIDLs like CSIDL_COMMON_PROGRAMS are
          passed to SHGetFolderPath on Windows 95, it actually returns
          "Windows\All Users\Start Menu\Programs", even though the Win95
          shell doesn't even look for a directory called "All Users".
            CSIDL_FLAG_CREATE is needed to avoid sparatic behavior of
          SHGetFolderPath (on Win95 at least) -- if it isn't used, it won't
          return certain paths at all (e.g. CSIDL_APPDATA), and some aren't
          correctly returned until a second call (e.g. CSIDL_PERSONAL).
            Also, note that SHGetFolderPath *seems* to have a slight memory
          leak in my tests; that is one of the reasons why "ShellFoldersRead"
          exists -- so that SHGetFolderPath is called only once per CSIDL. }
        if SUCCEEDED(SHGetFolderPathFunc(0, FolderIDs[Common, ID] or CSIDL_FLAG_CREATE,
           0, SHGFP_TYPE_CURRENT, Buf)) then
          Z := Buf;
      end;
      ShellFolders[Common, ID] := RemoveBackslashUnlessRoot(Z);
      ShellFoldersRead[Common, ID] := True;
    end;
    Result := ShellFolders[Common, ID];
  end;

begin
  Result := '';
  if not UsingWindows4 then
    Exit;
  if not IsAdmin then
    { If user isn't an administrator, always use User constants }
    Common := False;
  GetFolder (Common);
  if (Result = '') and Common then
    { If it failed to get the path of a Common CSIDL, try getting the
      User version of the CSIDL instead. (Many of the Common CSIDLS are
      unsupported by Win9x.) }
    GetFolder (False);
end;

function InstallOnThisVersion (const MinVersion: TSetupVersionData;
  const OnlyBelowVersion: TSetupVersionData): TInstallOnThisVersionResult;
var
  Ver, Ver2, MinVer, OnlyBelowVer: Cardinal;
begin
  Ver := WindowsVersion;
  if IsNT then begin
    MinVer := MinVersion.NTVersion;
    OnlyBelowVer := OnlyBelowVersion.NTVersion;
  end
  else begin
    MinVer := MinVersion.WinVersion;
    OnlyBelowVer := OnlyBelowVersion.WinVersion;
  end;
  Result := irInstall;
  if MinVer = 0 then
    Result := irNotOnThisPlatform
  else begin
    if (Ver < MinVer) or
       (IsNT and (LongRec(Ver).Hi = LongRec(MinVer).Hi) and
        (NTServicePackLevel < MinVersion.NTServicePack)) then
      Result := irVerTooLow
    else begin
      if OnlyBelowVer <> 0 then begin
        Ver2 := Ver;
        { A build number of 0 on OnlyBelowVersion means 'match any build' }
        if LongRec(OnlyBelowVer).Lo = 0 then
          Ver2 := Ver2 and $FFFF0000;  { set build number to zero on Ver2 also }
        if (Ver2 > OnlyBelowVer) or
           ((LongRec(Ver).Hi = LongRec(OnlyBelowVer).Hi) and
            (not IsNT or (NTServicePackLevel >= OnlyBelowVersion.NTServicePack))) then
          Result := irVerTooHigh;
      end;
    end;
  end;
end;

function GetSizeOfFiles (const Filename: String): Longint;
{ Returns sizes of all files matching Filename, or -1 if an error occured }
var
  H: THandle;
  FindData: TWin32FindData;
begin
  H := FindFirstFile(PChar(Filename), FindData);
  if H <> INVALID_HANDLE_VALUE then begin
    Result := 0;
    repeat
      if FindData.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY = 0 then begin
        if FindData.nFileSizeHigh <> 0 then begin  { file size > 4GB ! }
          Result := High(Longint);  { just return maximum size of a Longint (2GB) }
          Break;
        end
        else begin
          { If > 2GB (which is stored as a negative number on a signed 32-bit int),
            or if adding size to Result will cause it to wrap around (exceed 2GB),
            return 2GB }
          if (Longint(FindData.nFileSizeLow) < 0) or
             (Result + Longint(FindData.nFileSizeLow) < Result) then begin
            Result := High(Longint);
            Break;
          end;
          Inc (Result, FindData.nFileSizeLow);
        end;
      end;
    until not FindNextFile(H, FindData);
    WinProcs.FindClose (H);
  end
  else
    Result := -1;
end;

function GetSizeOfComponent(const ComponentName: String; const ExtraDiskSpaceRequired: Longint): Longint;
var
  FileComponents: TStringList;
  ComponentSize: LongInt;
  I: Integer;
begin
  ComponentSize := ExtraDiskSpaceRequired;
  FileComponents := TStringList.Create();

  for I := 0 to Entries[seFile].Count-1 do begin
    with PSetupFileEntry(Entries[seFile][I])^ do begin
      if (Tasks = '') and (Check = '') then begin {don't count tasks or scripted entries}
        FileComponents.CommaText := Components;
        if FileComponents.IndexOf(ComponentName) <> -1 then begin
          if LocationEntry <> -1 then
            Inc(ComponentSize, PSetupFileLocationEntry(Entries[seFileLocation][LocationEntry])^.OriginalSize)
          else
            Inc(ComponentSize, ExternalSize)
        end;
      end;
    end;
  end;

  FileComponents.Free();
  Result := ComponentSize;
end;

function GetSizeOfType(const TypeName: String; const IsCustom: Boolean): LongInt;
var
  ComponentTypes: TStringList;
  TypeSize: LongInt;
  I: Integer;
begin
  TypeSize := 0;
  ComponentTypes := TStringList.Create();

  for I := 0 to Entries[seComponent].Count-1 do begin
    with PSetupComponentEntry(Entries[seComponent][I])^ do begin
      ComponentTypes.CommaText := Types;
      { For custom types, only count fixed components. Otherwise count all. }
      if IsCustom then begin
        if (coFixed in Options) and (ComponentTypes.IndexOf(TypeName) <> -1) then
          Inc(TypeSize, Size);
      end else begin
        if ComponentTypes.IndexOf(TypeName) <> -1 then
          Inc(TypeSize, Size);
      end;
    end;
  end;

  ComponentTypes.Free();
  Result := TypeSize;
end;

procedure InternalError (const Id: String);
begin
  raise Exception.Create(FmtSetupMessage1(msgErrorInternal, Id));
end;

procedure InitializeSetup;
{ Initializes various vars used by the setup. This is called in the project
  source. }

  function VerToStr (Ver: Cardinal; ServicePack: Word): String;
  var
    Digits: Integer;
  begin
    with TSetupVersionDataVersion(Ver) do begin
      Digits := 2;
      if Minor mod 10 = 0 then begin
        Dec (Digits);
        Minor := Minor div 10;
      end;
      FmtStr (Result, '%d.%.*d', [Major, Digits, Minor]);
      if Build <> 0 then
        Result := Result + Format('.%d', [Build]);
      if ServicePack <> 0 then begin
        Result := Result + ' Service Pack ' + IntToStr(Hi(ServicePack));
        if Lo(ServicePack) <> 0 then
          Result := Result + Format('.%d', [Lo(ServicePack)]);
      end;
    end;
  end;

  procedure AbortInit (const Msg: TSetupMessageID);
  begin
    MsgBox (SetupMessages[Msg], '', mbCriticalError, MB_OK);
    Abort;
  end;

  procedure AbortInitFmt1 (const Msg: TSetupMessageID; const Arg1: String);
  begin
    MsgBox (FmtSetupMessage(Msg, [Arg1]), '', mbCriticalError, MB_OK);
    Abort;
  end;

  procedure AbortInitVerError (const Msg: TSetupMessageID; const Platform: String;
    const Ver: Cardinal; const ServicePack: Word);
  begin
    MsgBox (FmtSetupMessage(Msg, [Platform, VerToStr(Ver, ServicePack)]), '',
      mbCriticalError, MB_OK);
    Abort;
  end;

  procedure LoadSHFolderDLL;
  var
    H: HRSRC;
    Size: DWORD;
    R: HGLOBAL;
    P: Pointer;
    Filename: String;
    F: File;
    ExistingFileVersion, NewFileVersion: TFileVersionNumbers;
  const
    shfolder = 'shfolder.dll';
  begin
    {$R _shfoldr.res}  { Link in the .res file containing the DLL image }
    H := FindResource(0, 'SHFOLDERDLL', RT_RCDATA);
    if H = 0 then InternalError ('S100');
    Size := SizeofResource(0, H);
    if Size = 0 then InternalError ('S100');
    R := LoadResource(0, H);
    if R = 0 then InternalError ('S100');
    P := LockResource(R);
    if P = nil then InternalError ('S100');
    Filename := AddBackslash(TempInstallDir) + '_shfoldr.dll';
    AssignFile (F, Filename);
    FileMode := fmOpenReadWrite or fmShareExclusive;  Rewrite (F, 1);
    try
      BlockWrite (F, P^, Size);
    finally
      CloseFile (F);
    end;
    if not GetVersionNumbers(Filename, NewFileVersion) then
      InternalError ('S104');
    { Does the system already have the same version or a newer version of
      shfolder.dll? If so, use it instead of the one we just extracted. }
    if GetVersionNumbers(shfolder, ExistingFileVersion) and
       (((ExistingFileVersion.MS > NewFileVersion.MS) or
         ((ExistingFileVersion.MS = NewFileVersion.MS) and
          (ExistingFileVersion.LS > NewFileVersion.LS)))) or
        ((ExistingFileVersion.MS = NewFileVersion.MS) and
         (ExistingFileVersion.LS = NewFileVersion.LS)) then
      Filename := shfolder;
    { Ensure shell32.dll is pre-loaded so it isn't loaded/freed for each
      individual SHGetFolderPath call }
    SHFolderShell32Handle := LoadLibrary(shell32);
    SHFolderDLLHandle := LoadLibrary(PChar(Filename));
    if SHFolderDLLHandle = 0 then
      InternalError ('S105');
    @SHGetFolderPathFunc := GetProcAddress(SHFolderDLLHandle, 'SHGetFolderPathA');
    if @SHGetFolderPathFunc = nil then
      InternalError ('S106');
  end;

  procedure ReadWizardImage (var WizardImage: TBitmap; const Data: PDeflateBlockReadData);
  type
    PBuffer = ^TBuffer;
    TBuffer = array[0..8191] of Byte;
  var
    Buf: PBuffer;
    MemStream: TMemoryStream;
    BytesLeft, Bytes: Longint;
  begin
    Buf := nil;
    MemStream := nil;
    try
      New (Buf);
      MemStream := TMemoryStream.Create;
      InflateBlockRead (Data^, BytesLeft, SizeOf(BytesLeft));
      while BytesLeft > 0 do begin
        Bytes := BytesLeft;
        if Bytes > SizeOf(Buf^) then Bytes := SizeOf(Buf^);
        InflateBlockRead (Data^, Buf^, Bytes);
        MemStream.WriteBuffer (Buf^, Bytes);
        Dec (BytesLeft, Bytes);
      end;
      MemStream.Seek (0, soFromBeginning);
      WizardImage := TBitmap.Create;
      WizardImage.LoadFromStream (MemStream);
    finally
      MemStream.Free;
      if Assigned(Buf) then Dispose (Buf);
    end;
  end;

var
  PCount: Integer;

  SetupFilename: String;
  SetupFile: File;
  TestID: TSetupID;
  Data: PDeflateBlockReadData;
  BeganInflating: Boolean;

  I: Integer;

  Name: String;
  F: File;

  CurFileSize: Longint;
  OldErrorMode: UINT;

  procedure ReadEntries (const EntryType: TEntryType; const Count: Integer;
    const Size: Integer; const MinVersionOfs, OnlyBelowVersionOfs: Integer);
  var
    I: Integer;
    P: Pointer;
  begin
    for I := 1 to Count do begin
      P := AllocMem(Size);
      SEInflateBlockRead (Data^, P^, Size, EntryStrings[EntryType]);
      if InstallOnThisVersion(TSetupVersionData((@PByteArray(P)[MinVersionOfs])^),
         TSetupVersionData((@PByteArray(P)[OnlyBelowVersionOfs])^)) = irInstall then
        Entries[EntryType].Add (P)
      else
        SEFreeRec (P, EntryStrings[EntryType]);
    end;
  end;

var
  NewEntry: Pointer;
  MinimumTypeSpace: LongInt;
begin
  { Based on SetupLdr or not?
    Parameters for launching SetupLdr-based installation are:
    /SL3 <handle to SetupLdrWnd> <original exe filename> <setup msg data offset>
         <setup 0 data offset> <setup 1 data offset>
  }
  PCount := NewParamCount;
  if (PCount >= 6) and (NewParamStr(1) = '/SL3') then begin
    StartParam := 6;
    SetupLdrMode := True;
    SetupLdrWnd := StrToInt(NewParamStr(2));
    SetupLdrOriginalFilename := NewParamStr(3);
    SetupLdrOffsetMsg := StrToInt(NewParamStr(4));
    LoadCompressedSetupMessages (SetupLdrOriginalFilename, SetupLdrOffsetMsg,
      fmOpenRead or fmShareDenyWrite);
    SourceDir := ExtractFilePath(SetupLdrOriginalFilename);
    SetupLdrOffset0 := StrToInt(NewParamStr(5));
    SetupLdrOffset1 := StrToInt(NewParamStr(6));
  end
  else begin
    StartParam := 1;
    LoadCompressedSetupMessages (ExtractFilePath(ParamStr(0)) + 'SETUP.MSG', 0,
      fmOpenRead or fmShareDenyWrite);
    SetupLdrOriginalFilename := NewParamStr(0);
    SourceDir := ExtractFilePath(SetupLdrOriginalFilename);
  end;
  SourceDir := RemoveBackslashUnlessRoot(SourceDir);

  SetMessageBoxCaption (mbInformation, PChar(SetupMessages[msgInformationTitle]));
  SetMessageBoxCaption (mbConfirmation, PChar(SetupMessages[msgConfirmTitle]));
  SetMessageBoxCaption (mbError, PChar(SetupMessages[msgErrorTitle]));
  SetMessageBoxCaption (mbCriticalError, PChar(SetupMessages[msgErrorTitle]));

  if Win32Platform = VER_PLATFORM_WIN32s then
    AbortInitFmt1 (msgNotOnThisPlatform, 'Win32s');

  Application.Title := SetupMessages[msgSetupAppTitle];

  IsAdmin := IsAdminLoggedOn;
  Randomize;

  for I := StartParam to PCount do begin
    Name := NewParamStr(I);
    if CompareText(Name, '/Silent') = 0 then
      InitSilent := True
    else
    if CompareText(Name, '/VerySilent') = 0 then
      InitVerySilent := True
    else
    if CompareText(Name, '/NoRestart') = 0 then
      InitNoRestart := True
    else
    if CompareText(Name, '/NoIcons') = 0 then
      InitNoIcons := True
    else
    if CompareText(Copy(Name, 1, 12), '/Components=') = 0 then
      InitComponents.CommaText := Copy(Name, 13, Maxint)
    else
    if CompareText(Copy(Name, 1, 9), '/LoadInf=') = 0 then
      InitLoadInf := Copy(Name, 10, Maxint)
    else
    if CompareText(Copy(Name, 1, 9), '/SaveInf=') = 0 then
      InitSaveInf := Copy(Name, 10, Maxint)
    else
    if CompareText(Copy(Name, 1, 5), '/DIR=') = 0 then
      InitDir := Copy(Name, 6, Maxint)
    else
    if CompareText(Copy(Name, 1, 7), '/GROUP=') = 0 then
      InitProgramGroup := Copy(Name, 8, Maxint)
    else
    if CompareText(Name, '/DETACHEDMSG') = 0 then  { for debugging }
      DetachedUninstMsgFile := True;
  end;

  if InitLoadInf <> '' then
    LoadInf(InitLoadInf);

  { Read SETUP.0, or from EXE }
  if not SetupLdrMode then begin
    SetupFilename := AddBackslash(SourceDir) + 'SETUP.0'; {don't localize}
    if not FileExists(SetupFilename) then
      AbortInitFmt1 (msgSetupFileMissing, 'SETUP.0');
  end
  else
    SetupFilename := SetupLdrOriginalFilename;
  AssignFile (SetupFile, SetupFilename);
  FileMode := fmOpenRead or fmShareDenyWrite;  Reset (SetupFile, 1);
  try
    Seek (SetupFile, SetupLdrOffset0);
    BlockRead (SetupFile, TestID, SizeOf(TestID));
    if TestID <> SetupID then
      AbortInit (msgSetupFileCorruptOrWrongVer);
    try
      BeganInflating := False;
      New (Data);
      try
        InflateBlockReadBegin (SetupFile, Data^);
        BeganInflating := True;
        { Header }
        SEInflateBlockRead (Data^, SetupHeader, SizeOf(SetupHeader),
          SetupHeaderStrings);
        case InstallOnThisVersion(SetupHeader.MinVersion, SetupHeader.OnlyBelowVersion) of
          irInstall: ;
          irNotOnThisPlatform:
            if IsNT then
              AbortInitFmt1 (msgNotOnThisPlatform, 'Windows NT')
            else
              AbortInitFmt1 (msgOnlyOnThisPlatform, 'Windows NT');
          irVerTooLow:
            if IsNT then
              AbortInitVerError (msgWinVersionTooLowError, 'Windows NT',
                SetupHeader.MinVersion.NTVersion,
                SetupHeader.MinVersion.NTServicePack)
            else
              AbortInitVerError (msgWinVersionTooLowError, 'Windows',
                SetupHeader.MinVersion.WinVersion, 0);
          irVerTooHigh:
            if IsNT then
              AbortInitVerError (msgWinVersionTooHighError, 'Windows NT',
                SetupHeader.OnlyBelowVersion.NTVersion,
                SetupHeader.OnlyBelowVersion.NTServicePack)
            else
              AbortInitVerError (msgWinVersionTooHighError, 'Windows',
                SetupHeader.OnlyBelowVersion.WinVersion, 0);
        end;
        NeedPassword := shPassword in SetupHeader.Options;
        if (shAdminPrivilegesRequired in SetupHeader.Options) and
           not IsAdmin then
          AbortInit (msgAdminPrivilegesRequired);
        NeedsRestart := shAlwaysRestart in SetupHeader.Options;
        { Wizard image }
        ReadWizardImage (WizardImage, Data);
        ReadWizardImage (WizardSmallImage, Data);
        { Type entries }
        ReadEntries (seType, SetupHeader.NumTypeEntries, SizeOf(TSetupTypeEntry),
          Integer(@PSetupTypeEntry(nil).MinVersion),
          Integer(@PSetupTypeEntry(nil).OnlyBelowVersion));
        { Component entries }
        ReadEntries (seComponent, SetupHeader.NumComponentEntries, SizeOf(TSetupComponentEntry),
          Integer(@PSetupComponentEntry(nil).MinVersion),
          Integer(@PSetupComponentEntry(nil).OnlyBelowVersion));
        { Task entries }
        ReadEntries (seTask, SetupHeader.NumTaskEntries, SizeOf(TSetupTaskEntry),
          Integer(@PSetupTaskEntry(nil).MinVersion),
          Integer(@PSetupTaskEntry(nil).OnlyBelowVersion));
        { Dir entries }
        ReadEntries (seDir, SetupHeader.NumDirEntries, SizeOf(TSetupDirEntry),
          Integer(@PSetupDirEntry(nil).MinVersion),
          Integer(@PSetupDirEntry(nil).OnlyBelowVersion));
        { File entries }
        ReadEntries (seFile, SetupHeader.NumFileEntries, SizeOf(TSetupFileEntry),
          Integer(@PSetupFileEntry(nil).MinVersion),
          Integer(@PSetupFileEntry(nil).OnlyBelowVersion));
        { Icon entries }
        ReadEntries (seIcon, SetupHeader.NumIconEntries, SizeOf(TSetupIconEntry),
          Integer(@PSetupIconEntry(nil).MinVersion),
          Integer(@PSetupIconEntry(nil).OnlyBelowVersion));
        { INI entries }
        ReadEntries (seIni, SetupHeader.NumIniEntries, SizeOf(TSetupIniEntry),
          Integer(@PSetupIniEntry(nil).MinVersion),
          Integer(@PSetupIniEntry(nil).OnlyBelowVersion));
        { Registry entries }
        ReadEntries (seRegistry, SetupHeader.NumRegistryEntries, SizeOf(TSetupRegistryEntry),
          Integer(@PSetupRegistryEntry(nil).MinVersion),
          Integer(@PSetupRegistryEntry(nil).OnlyBelowVersion));
        { InstallDelete entries }
        ReadEntries (seInstallDelete, SetupHeader.NumInstallDeleteEntries, SizeOf(TSetupDeleteEntry),
          Integer(@PSetupDeleteEntry(nil).MinVersion),
          Integer(@PSetupDeleteEntry(nil).OnlyBelowVersion));
        { UninstallDelete entries }
        ReadEntries (seUninstallDelete, SetupHeader.NumUninstallDeleteEntries, SizeOf(TSetupDeleteEntry),
          Integer(@PSetupDeleteEntry(nil).MinVersion),
          Integer(@PSetupDeleteEntry(nil).OnlyBelowVersion));
        { Run entries }
        ReadEntries (seRun, SetupHeader.NumRunEntries, SizeOf(TSetupRunEntry),
          Integer(@PSetupRunEntry(nil).MinVersion),
          Integer(@PSetupRunEntry(nil).OnlyBelowVersion));
        { UninstallRun entries }
        ReadEntries (seUninstallRun, SetupHeader.NumUninstallRunEntries, SizeOf(TSetupRunEntry),
          Integer(@PSetupRunEntry(nil).MinVersion),
          Integer(@PSetupRunEntry(nil).OnlyBelowVersion));

        BeganInflating := False;
        InflateBlockReadEnd (Data^);

        InflateBlockReadBegin (SetupFile, Data^);
        BeganInflating := True;

        { File location entries }
        for I := 1 to SetupHeader.NumFileLocationEntries do begin
          GetMem (NewEntry, SizeOf(TSetupFileLocationEntry));
          InflateBlockRead (Data^, NewEntry^, SizeOf(TSetupFileLocationEntry));
          Entries[seFileLocation].Add (NewEntry);
        end;
      finally
        if BeganInflating then
          InflateBlockReadEnd (Data^);
        Dispose (Data);
      end;
    except
      on EZlibDataError do
        AbortInit (msgSetupFileCorrupt);
    end;
  finally
    CloseFile (SetupFile);
  end;

  { Setup install mode }
  if InitSilent then
    SetupHeader.InstallMode := imSilent
  else if InitVerySilent then
    SetupHeader.InstallMode := imVerySilent;

  if SetupHeader.InstallMode <> imNormal then begin
    if SetupHeader.InstallMode = imVerySilent then
      Application.ShowMainForm := False;
    SetupHeader.Options := SetupHeader.Options - [shWindowVisible];
  end;

  { Check if app is running }
  while CheckForMutexes(SetupHeader.AppMutex) do
    if MsgBox(FmtSetupMessage1(msgSetupAppRunningError, SetupHeader.AppName),
       SetupMessages[msgSetupAppTitle], mbError, MB_OKCANCEL) <> IDOK then
      Abort;

  { Read Windows dir }
  WinDir := GetWinDir;

  { Read Windows System dir. If it is a shared version it uses Windows dir
    instead. }
  WinSystemDir := GetSystemDir;
  if Pos(WinDir, WinSystemDir) <> 1 then begin
    { If System is not a subdirectory of Windows, possibly a shared version }
    try
      { Try making a temp file to see if the dir is write protected }
      Name := GenerateUniqueName(WinSystemDir, '.tmp');
      AssignFile (F, Name);
      FileMode := fmOpenWrite or fmShareExclusive;  Rewrite (F, 1);
      CloseFile (F);
      DeleteFile (Name);
    except
      { If exception occured, system dir is probably write protected. }
      WinSystemDir := WinDir;
    end;
  end;

  { Get system drive }
  if Win32Platform = VER_PLATFORM_WIN32_NT then
    SystemDrive := GetEnv('SystemDrive')  {don't localize}
  else
    SystemDrive := '';
  if SystemDrive = '' then begin
    SystemDrive := ExtractFileDrive(WinDir);
    if SystemDrive = '' then
      { In some rare case that ExtractFileDrive failed, just default to C }
      SystemDrive := 'C:';
  end;

  { Attempt to create a TempInstallDir }
  Name := GenerateUniqueName(GetTempDir, '.tmp');
  if not CreateDirectory(PChar(Name), nil) then
    raise Exception.Create(FmtSetupMessage1(msgErrorCreatingDir, Name));
  TempInstallDir := Name;

  { Extract "_shfoldr.dll" to TempInstallDir, and load it }
  if UsingWindows4 then
    LoadSHFolderDLL;

  { Get Program Files and Common Files dirs }
  ProgramFilesDir := GetProgramFilesPath;
  if ProgramFilesDir = '' then
    ProgramFilesDir := SystemDrive + '\Program Files';  {don't localize}
  CommonFilesDir := GetCommonFilesPath;
  if CommonFilesDir = '' then
    CommonFilesDir := AddBackslash(ProgramFilesDir) + 'Common Files';  {don't localize}

  { Get Fonts directory }
  FontDir := GetShellFolder(False, sfFonts);
  if FontDir = '' then
    FontDir := WinSystemDir;

  { Generate DAO directory name }
  if UsingWindows4 then
    DAODir := AddBackslash(CommonFilesDir) + 'Microsoft Shared\DAO'
  else
    DAODir := AddBackslash(WinDir) + 'MSAPPS\DAO';

  { Setup script }
  HasCodeText := SetupHeader.CodeText <> '';
  if HasCodeText then begin
    ScriptInit(SetupHeader.CodeText);
    if ScriptRunBooleanFunction('InitializeSetup', [''], False, True) = False then
      Abort;
  end;

  { Remove types that fail their 'check'. Can't do this earlier because the
    InitializeSetup call above can't be done earlier. }
  for I := 0 to Entries[seType].Count-1 do begin
    if not ShouldProcessEntry(nil, nil, '', '', PSetupTypeEntry(Entries[seType][I]).Check) then begin
      SEFreeRec(Entries[seType][I], EntryStrings[seType]);
      { Don't delete it yet so that the entries can be processed sequentially }
      Entries[seType][I] := nil;
    end;
  end;
  { Delete the nil-ed items now }
  Entries[seType].Pack();

  { Remove components }
  for I := 0 to Entries[seComponent].Count-1 do begin
    if not ShouldProcessEntry(nil, nil, '', '', PSetupComponentEntry(Entries[seComponent][I]).Check) then begin
      SEFreeRec(Entries[seComponent][I], EntryStrings[seComponent]);
      Entries[seComponent][I] := nil;
    end;
  end;
  Entries[seComponent].Pack();

  { Remove tasks }
  for I := 0 to Entries[seTask].Count-1 do begin
    if not ShouldProcessEntry(nil, nil, '', '', PSetupTaskEntry(Entries[seTask][I]).Check) then begin
      SEFreeRec(Entries[seTask][I], EntryStrings[seTask]);
      Entries[seTask][I] := nil;
    end;
  end;
  Entries[seTask].Pack();

  { Set misc. variables }
  UninstallRegKeyBaseName := SetupHeader.AppId;
  { Uninstall registry keys can only be up to 63 characters, otherwise Win95
    ignores them. Limit to 57 since Setup will add _isXXX to the end later. }
  if Length(UninstallRegKeyBaseName) > 57 then
    { Only keep the first 48 characters, then add an tilde and the CRC
      of the original string (to make the trimmed string unique). The
      resulting string is 57 characters long. }
    FmtStr (UninstallRegKeyBaseName, '%.48s~%.8x', [UninstallRegKeyBaseName,
      GetCRC32(UninstallRegKeyBaseName[1], Length(UninstallRegKeyBaseName))]);

  HasCustomType := False;
  for I := 0 to Entries[seType].Count-1 do begin
    if toIsCustom in PSetupTypeEntry(Entries[seType][I]).Options then begin
      HasCustomType := True;
      Break;
    end;
  end;

  HasComponents := Entries[seComponent].Count <> 0;

  HasIcons := (Entries[seIcon].Count <> 0) or ((shUninstallable in SetupHeader.Options)
    and ((shAlwaysCreateUninstallIcon in SetupHeader.Options) or not UsingWindows4));

  HasTasklessIcons := False;
  HasTaskIcons := False;
  for I := 0 to Entries[seIcon].Count-1 do begin
    if PSetupIconEntry(Entries[seIcon][I]).Tasks = '' then
      HasTasklessIcons := True
    else
      HasTaskIcons := True;
    if HasTasklessIcons and HasTaskIcons then
      Break;
  end;

  HasTasks := Entries[seTask].Count <> 0;

  { Calculate minimum disk space. If there are setup types, find the smallest
    type and add the size of all files that don't belong to any component. Otherwise
    calculate minimum disk space by adding all of the file's sizes. Also for each
    "external" file, check the file size now, and store it the ExternalSize field
    of the TSetupFileEntry record. }

  OldErrorMode := SetErrorMode(SEM_FAILCRITICALERRORS);  { Prevent "Network Error" boxes }
  try
    MinimumSpace := SetupHeader.ExtraDiskSpaceRequired;

    for I := 0 to Entries[seFile].Count-1 do begin
      with PSetupFileEntry(Entries[seFile][I])^ do begin
        if LocationEntry <> -1 then begin { not an "external" file }
          if Components = '' then { no types or a file that doesn't belong to any component }
            if (Tasks = '') and (Check = '') then {don't count tasks and scripted entries}
              Inc(MinimumSpace, PSetupFileLocationEntry(Entries[seFileLocation][LocationEntry])^.OriginalSize)
        end else begin
          try
            CurFileSize := GetSizeOfFiles(ChangeDirConst(SourceFilename));
          except
            { ignore exceptions }
            CurFileSize := 0;
          end;
          if CurFileSize < 0 then
            CurFileSize := 0;
          ExternalSize := CurFileSize;
          if Components = '' then { no types or a file that doesn't belong to any component }
            if (Tasks = '') and (Check = '') then {don't count tasks or scripted entries}
              Inc (MinimumSpace, CurFileSize);
        end;
      end;
    end;

    for I := 0 to Entries[seComponent].Count-1 do
      with PSetupComponentEntry(Entries[seComponent][I])^ do
        Size := GetSizeOfComponent(Name, ExtraDiskSpaceRequired);

    if Entries[seType].Count > 0 then begin
      MinimumTypeSpace := -1;
      for I := 0 to Entries[seType].Count-1 do begin
        with PSetupTypeEntry(Entries[seType][I])^ do begin
          Size := GetSizeOfType(Name, toIsCustom in Options);
          if (MinimumTypeSpace = -1) or (Size < MinimumTypeSpace) then
            MinimumTypeSpace := Size;
        end;
      end;
      Inc(MinimumSpace, MinimumTypeSpace);
    end;

  finally
    SetErrorMode (OldErrorMode);
  end;

end;

procedure DeinitSetup;
var
  I: Integer;
begin
  if HasCodeText then
    ScriptRunProcedure('DeInitializeSetup', [''], False);

  for I := 0 to DeleteFilesAfterInstallList.Count-1 do
    DeleteFile (DeleteFilesAfterInstallList[I]);
  DeleteFilesAfterInstallList.Clear;
  for I := 0 to DeleteDirsAfterInstallList.Count-1 do
    RemoveDirectory (PChar(DeleteDirsAfterInstallList[I]));
  DeleteDirsAfterInstallList.Clear;

  { Free the shfolder.dll handles }
  if SHFolderDLLHandle <> 0 then begin
    @SHGetFolderPathFunc := nil;
    FreeLibrary (SHFolderDLLHandle);
  end;
  if SHFolderShell32Handle <> 0 then
    FreeLibrary (SHFolderShell32Handle);

  if TempInstallDir <> '' then
    DelTree (TempInstallDir, True, True, True, nil, nil);

  if RestartSystem then begin
    if SetupLdrMode then
      { Send a special message back to SetupLdr telling it to restart the system
        after Setup returns }
      SendMessage (SetupLdrWnd, WM_USER + 150, 10000, 0)
    else
      { Restart now if we're not being run from SetupLdr.
        Note: RestartComputer doesn't return if successful. }
      RestartComputer;
  end;
end;

function BlendRGB (const Color1, Color2: TColor; const Blend: Integer): TColor;
{ Blends Color1 and Color2. Blend must be between 0 and 63; 0 = all Color1,
  63 = all Color2. }
type
  TColorBytes = array[0..3] of Byte;
var
  I: Integer;
begin
  Result := 0;
  for I := 0 to 2 do
    TColorBytes(Result)[I] := Integer(TColorBytes(Color1)[I] +
      ((TColorBytes(Color2)[I] - TColorBytes(Color1)[I]) * Blend) div 63);
end;

function ExitSetupMsgBox: Boolean;
begin
  Result := MsgBox(SetupMessages[msgExitSetupMessage], SetupMessages[msgExitSetupTitle],
    mbConfirmation, MB_YESNO or MB_DEFBUTTON2) = ID_YES;
end;

procedure SetFontNameSize (const AFont: TFont; const ANameSize: String;
  const ADefaultName: String; const ADefaultSize: Integer);
var
  N: String;
  S: Integer;
begin
  N := ADefaultName;
  S := ADefaultSize;
  ExtractFontNameSize (ANameSize, N, S);
  if S >= 0 then  { points (positive) }
    S := MulDiv(-S, Screen.PixelsPerInch, 72)
  else  { pixels (negative) }
    S := MulDiv(S, Screen.PixelsPerInch, 96);

  AFont.Name := N;
  AFont.Height := S;
end;

type
  TFormAccess = class(TForm);

procedure SetFormFont (Form: TForm; var OldTextHeight, NewTextHeight: Integer);
var
  W, H: Integer;
  R: TRect;
begin
  with Form do begin
    OldTextHeight := 13;
    SetFontNameSize (Font, SetupMessages[msg_DialogFont], 'MS Sans Serif', 8);
    NewTextHeight := Canvas.TextHeight('0');
    if NewTextHeight <> OldTextHeight then begin
      { Loosely based on scaling code from TForm.ReadState: }
      TFormAccess(Form).ScaleControls (NewTextHeight, OldTextHeight);
      R := ClientRect;
      W := MulDiv(R.Right, NewTextHeight, OldTextHeight);
      H := MulDiv(R.Bottom, NewTextHeight, OldTextHeight);
      SetBounds (Left, Top, W + (Width - R.Right), H + (Height - R.Bottom));
    end;
  end;
end;


{ TMainForm }

constructor TMainForm.Create (AOwner: TComponent);
var
  SystemMenu: HMenu;
  Palette: packed record
    palVersion: Word;
    palNumEntries: Word;
    palPalEntry: array[0..63] of TPaletteEntry;
  end;
  Entry: Integer;
  C1, C2: TColor;
begin
  inherited;

  if shWindowVisible in SetupHeader.Options then begin
    { Should the main window not be sizable? }
    if not(shWindowShowCaption in SetupHeader.Options) then
      BorderStyle := bsNone
    else
    if not(shWindowResizable in SetupHeader.Options) then
      BorderStyle := bsSingle;

    { Make the main window full-screen }
    SetBounds (0, 0, GetSystemMetrics(SM_CXSCREEN), GetSystemMetrics(SM_CYSCREEN));
    { Before maximizing the window, ensure Handle is created now so the correct
      'restored' position is saved properly }
    HandleNeeded;

    { Maximize the window so that the taskbar is still accessible }
    if shWindowStartMaximized in SetupHeader.Options then
      WindowState := wsMaximized;
  end
  else begin
    BorderStyle := bsNone;
    SetBounds (0, 0, 0, 0);
  end;

  Caption := FmtSetupMessage1(msgSetupWindowTitle, SetupHeader.AppName);

  { Append the 'About Setup' item to the system menu }
  SystemMenu := GetSystemMenu(Handle, False);
  AppendMenu (SystemMenu, MF_SEPARATOR, 0, nil);
  AppendMenu (SystemMenu, MF_STRING, 9999, PChar(SetupMessages[msgAboutSetupMenuItem]));

  if SetupHeader.BackColor <> SetupHeader.BackColor2 then begin
    { Create the palette }
    with Palette do begin
      palVersion := $300;
      palNumEntries := 64;
    end;
    C1 := ColorToRGB(SetupHeader.BackColor);
    C2 := ColorToRGB(SetupHeader.BackColor2);
    for Entry := 0 to 63 do
      { rest of entries }
      TColor(Palette.palPalEntry[Entry]) := BlendRGB(C1, C2, Entry);
    NewPalette := CreatePalette(PLogPalette(@Palette)^);
  end;

  Application.OnActivate := AppOnActivate;
  Application.HookMainWindow (MainWindowHook);
  Application.OnRestore := AppOnRestore;

  { Very silent installs totally hide the main window, so we've go to 'activate' it ourselves }
  if SetupHeader.InstallMode = imVerySilent then
    FormActivate(Self);
end;

destructor TMainForm.Destroy;
begin
  Application.UnhookMainWindow (MainWindowHook);
  inherited;
  if NewPalette <> 0 then
    DeleteObject (NewPalette);
end;

procedure TMainForm.CenterForm (AForm: TForm);
var
  R: TRect;
  X, Y: Integer;
begin
  if shWindowVisible in SetupHeader.Options then begin
    R := ClientRect;
    MapWindowPoints (Handle, 0, R, 2);
  end
  else
    R := Rect(0, 0, GetSystemMetrics(SM_CXSCREEN), GetSystemMetrics(SM_CYSCREEN));
  X := ((R.Right+R.Left) div 2)-(AForm.Width div 2);
  if X < 0 then X := 0;
  Y := ((R.Bottom+R.Top) div 2)-(AForm.Height div 2);
  if Y < 0 then Y := 0;
  AForm.SetBounds (X, Y, AForm.Width, AForm.Height);
end;

procedure TMainForm.WMSysCommand (var Message: TWMSysCommand);
begin
  if (Message.CmdType = 9999) and not InExecuteLoop then
    ShowAboutBox
  else
    case Message.CmdType of
      SC_MOVE, SC_MOVE + 2:Message.Result := 0;
      SC_SIZE..SC_SIZE + 8:Message.Result := 0;
    else
      inherited;
    end;
end;

function TMainForm.GetPalette: HPALETTE;
begin
  Result := NewPalette;
end;

procedure TMainForm.AppOnActivate(Sender: TObject);
begin
  { This corrects some rare palette problems }
  if NewPalette <> 0 then
    Repaint;
end;

procedure TMainForm.WMEraseBkgnd (var Message: TWMEraseBkgnd);
begin
  { Since the form paints its entire client area in FormPaint, there is
    no need for the VCL to ever erase the client area with the brush color.
    Doing so only slows it down, so this message handler disables that default
    behavior. }
  Message.Result := 0;
end;

procedure TMainForm.FormPaint(Sender: TObject);

  function IsMultiByteString(s: string): Boolean;
  var
    i: integer;
  begin
    for i := 1 to Length(s) do
      if ByteType(s, i) <> mbSingleByte then Break;
    Result := (i <= Length(s));
  end;

var
  C1, C2: TColor;
  CS: TPoint;
  Z: Integer;
  R, R2: TRect;
begin
  with Canvas do begin
    { Draw the blue background }
    if NewPalette <> 0 then
      SelectPalette (Handle, NewPalette, False);
    if SetupHeader.BackColor = SetupHeader.BackColor2 then begin
      Brush.Color := SetupHeader.BackColor;
      FillRect (ClientRect);
    end
    else begin
      C1 := ColorToRGB(SetupHeader.BackColor);
      C2 := ColorToRGB(SetupHeader.BackColor2);
      CS := ClientRect.BottomRight;
      for Z := 0 to 63 do begin
        Brush.Color := BlendRGB(C1, C2, Z) or $02000000;
        if not(shBackColorHorizontal in SetupHeader.Options) then
          FillRect (Rect(0, MulDiv(CS.Y, Z, 63), CS.X, MulDiv(CS.Y, Z+1, 63)))
        else
          FillRect (Rect(MulDiv(CS.X, Z, 63), 0, MulDiv(CS.X, Z+1, 63), CS.Y));
      end;
    end;

    { Draw the application name and copyright }
    SetBkMode (Handle, TRANSPARENT);

    SetFontNameSize (Font, SetupMessages[msg_TitleFont], 'Arial', 29);
    if IsMultiByteString(SetupHeader.AppName) then
      Font.Style := [fsBold]
    else
      Font.Style := [fsBold, fsItalic];
    R := ClientRect;
    InflateRect (R, -8, -8);
    R2 := R;
    OffsetRect (R2, 4, 4);
    Font.Color := clBlack;
    DrawText (Handle, PChar(SetupHeader.AppName), -1, R2, DT_TOP or DT_LEFT or
      DT_WORDBREAK or DT_NOPREFIX);
    Font.Color := clWhite;
    DrawText (Handle, PChar(SetupHeader.AppName), -1, R, DT_TOP or DT_LEFT or
      DT_WORDBREAK or DT_NOPREFIX);

    SetFontNameSize (Font, SetupMessages[msg_CopyrightFont], 'Arial', 8);
    Font.Style := [];
    R := ClientRect;
    InflateRect (R, -6, -6);
    R2 := R;
    DrawText (Handle, PChar(SetupHeader.AppCopyright), -1, R2, DT_TOP or DT_RIGHT or
      DT_WORDBREAK or DT_NOPREFIX or DT_CALCRECT);
    R.Top := R.Bottom - (R2.Bottom - R2.Top);
    R2 := R;
    OffsetRect (R2, 1, 1);
    Font.Color := clBlack;
    DrawText (Handle, PChar(SetupHeader.AppCopyright), -1, R2, DT_TOP or DT_RIGHT or
      DT_WORDBREAK or DT_NOPREFIX);
    Font.Color := clWhite;
    DrawText (Handle, PChar(SetupHeader.AppCopyright), -1, R, DT_TOP or DT_RIGHT or
      DT_WORDBREAK or DT_NOPREFIX);
  end;
end;

procedure TMainForm.FormResize(Sender: TObject);
begin
  { Needs to redraw the background whenever the form is resized }
  Repaint;
end;

procedure TMainForm.ShowAboutBox;
const
  { Removing the About box or modifying the text inside it is a violation of the
    Inno Setup license agreement; see LICENSE.TXT. However, adding additional
    lines to the About box is permitted. }
  DefAboutText = '%1 version %2' + SNewLine + '%3' + SNewLine2 +
    '%1 home page:' + SNewLine + '%4';
  AboutTextArg1 = 'Inno Setup';
  AboutTextArg2 = SetupVersion;
  AboutTextArg3 = 'Copyright (C) 1998-2000 Jordan Russell. All rights reserved.';
  AboutTextArg4 = 'http://www.innosetup.com/';
var
  P, Z: PChar;
  I: Integer;
  S: String;
begin
  P := PChar(SetupMessages[msgAboutSetupMessage]);
  I := 0;
  Z := P;
  while Z^ <> #0 do begin
    if Z^ = '%' then begin
      Inc (Z);
      if Z^ = #0 then Break;
      if Z^ in ['1'..'9'] then
        I := I or (1 shl (Ord(Z^) - Ord('1')));
    end;
    Inc (Z);
  end;
  if I and 15 <> 15 then
    P := DefAboutText;
  S := FmtMessage(P, [AboutTextArg1, AboutTextArg2, AboutTextArg3,
    AboutTextArg4]);
  if SetupMessages[msgAboutSetupNote] <> '' then
    S := S + SNewLine2 + SetupMessages[msgAboutSetupNote];
  MsgBox (S, SetupMessages[msgAboutSetupTitle], mbInformation, MB_OK);
end;

procedure ProcessMessagesProc; far;
begin
  Application.ProcessMessages;
end;

procedure TMainForm.WMSetupNextStep (var Msg: TMessage);

  procedure ProcessRunEntry(const RunEntry: PSetupRunEntry; var ErrorCode: Integer);
  var
    S: String;
  begin
    with RunEntry^ do begin
      S := ChangeDirConst(Name);
      if not(roShellExec in Options) then begin
        if (not(roSkipIfDoesntExist in Options) or FileExists(S)) and
           not InstExec(S, ChangeDirConst(Parameters),
             ChangeDirConst(WorkingDir), Wait = rwWaitUntilTerminated,
             Wait = rwWaitUntilIdle, ShowCmd, ProcessMessagesProc,
             ErrorCode) then
          MsgBox (FmtSetupMessage1(msgErrorExecutingProgram, S) + SNewLine2 +
            FmtSetupMessage(msgErrorFunctionFailedWithMessage, ['CreateProcess',
              IntToStr(ErrorCode), SysErrorMessage(ErrorCode)]),
            '', mbCriticalError, MB_OK);
      end
      else begin
        if (not(roSkipIfDoesntExist in Options) or FileOrDirExists(S)) and
           not InstShellExec(S, ChangeDirConst(Parameters),
             ChangeDirConst(WorkingDir), ShowCmd, ErrorCode) then
          MsgBox (FmtSetupMessage1(msgErrorExecutingProgram, S) + SNewLine2 +
            FmtSetupMessage(msgErrorFunctionFailed, ['ShellExecute',
              IntToStr(ErrorCode)]),
            '', mbCriticalError, MB_OK);
      end;
    end;
  end;

var
  I, ErrorCode: Integer;
  RunEntry: PSetupRunEntry;
  CheckBox: TCheckBox;
  DisableFinishedPage: Boolean;
label NextStep;
begin
  try
    NextStep:
    { Finish up the previous step }
    case CurStep of
      csStart: ;
      csWizard: begin
          WizardDirValue := WizardForm.DirEdit.Text;
          WizardGroupValue := WizardForm.GroupEdit.Text;
          WizardNoIcons := WizardForm.NoIconsCheck.Checked;
          WizardSetupType := WizardForm.GetSetupType();
          WizardForm.GetSelectedComponents(WizardComponents, False);
          WizardForm.GetSelectedTasks(WizardTasks, False);
          if InitSaveInf <> '' then
            SaveInf(InitSaveInf);
          { Make sure that riched*.dll is unloaded before installation is
            started, so that installations can replace it. }
          WizardForm.LicenseMemo.Text := '';
          WizardForm.LicenseMemo.UseRichEdit := False;
          WizardForm.InfoBeforeMemo.Text := '';
          WizardForm.InfoBeforeMemo.UseRichEdit := False;
        end;
      csCopy: begin
          InstallForm.Free;
          InstallForm := nil;
          if Entries[seRun].Count <> 0 then begin
            InExecuteLoop := True;
            try
              for I := 0 to Entries[seRun].Count-1 do begin
                RunEntry := PSetupRunEntry(Entries[seRun][I]);
                if ShouldProcessRunEntry(WizardComponents, WizardTasks, RunEntry) and not(roPostInstall in RunEntry.Options) then begin
                  {$IFNDEF WIZARDCLASSIC}
                    if WizardForm.Visible then
                      WizardForm.Hide();
                  {$ENDIF}
                  ProcessRunEntry(RunEntry, ErrorCode);
                end;
              end;
            finally
              InExecuteLoop := False;
            end;
            Application.BringToFront;
          end;
        end;
      csFinished: begin
          if not NeedsRestart then begin
            InExecuteLoop := True;
            try
              for I := 0 to RunCheckBoxes.Count-1 do begin
                CheckBox := TCheckBox(RunCheckboxes[I]);
                if CheckBox.Checked then begin
                  RunEntry := PSetupRunEntry(CheckBox.Tag);
                  ProcessRunEntry(RunEntry, ErrorCode);
                end;
              end;
            finally
              InExecuteLoop := False;
            end;
            RunCheckboxes.Free();
          end else begin
            case SetupHeader.InstallMode of
              imNormal:
                RestartSystem := WizardForm.YesRadio.Checked;
              imSilent:
                RestartSystem := not InitNoRestart and (MsgBox (FixLabel(SetupMessages[msgFinishedRestartMessage], MinimumSpace), '', mbConfirmation, MB_YESNO) = IDYES);
              imVerySilent:
                RestartSystem := not InitNoRestart;
            end;
          end;
        end;
    end;

    { Proceed to next step }
    Inc (CurStep);
    if HasCodeText then
      ScriptRunProcedure('CurStepChanged', ['CurStep', Integer(CurStep)], False);

    case CurStep of
      csWizard: begin
            Application.Restore;
            WizardForm := TWizardForm.Create(Application);
            if SetupHeader.InstallMode <> imNormal then begin
              {$IFNDEF WIZARDCLASSIC}
              if SetupHeader.InstallMode = imSilent then begin
                WizardForm.CurPage := wpInstalling;
                WizardForm.CurPageChanged;
                WizardForm.Show();
              end;
              {$ENDIF}
              goto NextStep;
            end else
              WizardForm.Show();
          end;
      csCopy: begin
            Application.Restore;
            Update;
            InstallForm := TInstallForm.Create(Application);
            if SetupHeader.InstallMode <> imVerySilent then begin
              {$IFDEF WIZARDCLASSIC}
                InstallForm.Show;
              {$ELSE}
                InstallForm.BorderStyle := bsNone;
                InstallForm.CancelButton.Visible := False;
                InstallForm.Parent := WizardForm.InstallingPanel;
                { Make InstallForm fill InstallingPanel }
                with InstallForm do
                  SetBounds (0, 0, WizardForm.InstallingPanel.Width, WizardForm.InstallingPanel.Height);
                { Extend the labels & progress meter to fill the width of InstallingPanel }
                with InstallForm.StatusLabel do
                  SetBounds (0, Top, InstallForm.Width, Height);
                with InstallForm.FilenameLabel do
                  SetBounds (0, Top, InstallForm.Width, Height);
                with InstallForm.ProgressGauge do
                  SetBounds (0, Top, InstallForm.Width, Height);
                //InstallForm.Show;   ... don't need Show's BringToFront call
                InstallForm.Visible := True;
              {$ENDIF}
            end;
            InstallForm.StartInstalling;
          end;
      csFinished: begin
            if NeedsRestart then begin
              with WizardForm do begin
                ChangeFinishedLabel (FixLabel(SetupMessages[msgFinishedRestartLabel], MinimumSpace));
                YesRadio.Visible := True;
                NoRadio.Visible := True;
              end;
            end else begin
              RunCheckBoxes := TList.Create();
              WizardForm.CreateRunButtons(WizardComponents, WizardTasks, RunCheckBoxes);
            end;

            if SetupHeader.InstallMode <> imNormal then
              goto NextStep;

            if SetupHeader.InfoAfterText = '' then begin
              DisableFinishedPage := not NeedsRestart and (shDisableFinishedPage in SetupHeader.Options);
              if not DisableFinishedPage then begin
                if HasCodeText then
                  DisableFinishedPage := ScriptRunBooleanFunction('SkipCurPage', ['CurPage', Ord(wpFinished)], False, DisableFinishedPage);
              end;
              if DisableFinishedPage then
                goto NextStep;  { proceed to 'csTerminate' step }
              WizardForm.CurPage := wpFinished;
            end
            else begin
              { Don't set up the rich-edit InfoAfterMemo until now, so that
                installations can replace riched*.dll }
              WizardForm.InfoAfterMemo.UseRichEdit := True;
              WizardForm.InfoAfterMemo.RTFText := SetupHeader.InfoAfterText;
              WizardForm.CurPage := wpInfoAfter;
            end;
            Application.Restore;
            Update;
            WizardForm.CurPageChanged;
            WizardForm.CantShow := False;
            WizardForm.Show;
          end;
      csTerminate: Close;
    end;
  except
    { If an exception was raised, display the message, then terminate }
    Application.HandleException (Self);
    CurStep := csTerminate;
    if HasCodeText then
      ScriptRunProcedure('CurStepChanged', ['CurStep', Integer(CurStep)], False);
    Close;
  end;
end;

procedure TMainForm.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  if not InExecuteLoop then begin
    if (CurStep = csCopy) and Assigned(InstallForm) then begin
      CanClose := False;
      InstallForm.Close;
    end
    else
      if CurStep < csFinished then
        CanClose := ExitSetupMsgBox
      else
        CanClose := True;
  end
  else
    CanClose := False;
end;

procedure TMainForm.FormActivate(Sender: TObject);
const
  StartedSteps: Boolean = False;
begin
  { Start the 'steps' }
  if not StartedSteps then begin
    CurStep := csStart;
    if HasCodeText then
      ScriptRunProcedure('CurStepChanged', ['CurStep', Integer(CurStep)], False);
    Perform (WM_SETUPNEXTSTEP, 0, 0);
    StartedSteps := True;
  end;
end;

procedure TMainForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  Closing := True;
end;

procedure TMainForm.WMGetDlgCode (var Message: TWMGetDlgCode);
begin
  Message.Result := Message.Result or DLGC_WANTTAB;
end;

function EWP (Wnd: HWND; Param: Longint): BOOL; stdcall;
begin
  { Note: GetParent is not used here because the other windows are not
    actually child windows since they don't have WS_CHILD set. }
  if GetWindowLong(Wnd, GWL_HWNDPARENT) <> Param then
    Result := True
  else begin
    Result := False;
    BringWindowToTop (Wnd);
  end;
end;

procedure TMainForm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  { If, for some reason, the user doesn't have a mouse and the main form was
    activated, there would normally be no way to reactivate the child form.
    But this reactivates the form if the user hits a key on the keyboard }
  Key := 0;
  EnumWindows (@EWP, Handle);
end;

type
  PEWPRec = ^TEWPRec;
  TEWPRec = record
    Window: HWND;
    NotMinimized: Boolean;
  end;
function EWP2 (Wnd: HWND; Param: PEWPRec): BOOL; stdcall;
var
  Ctl: TWinControl;
begin
  Result := True;
  { Note: GetParent is not used here because the other windows are not
    actually child windows since they don't have WS_CHILD set. }
  if HWND(GetWindowLong(Wnd, GWL_HWNDPARENT)) = Param^.Window then begin
    Ctl := FindControl(Wnd);
    if Ctl is TSetupChildForm then
      with TSetupChildForm(Ctl) do
        Visible := Param^.NotMinimized and not CantShow;
  end;
end;

function TMainForm.MainWindowHook (var Message: TMessage): Boolean;
var
  E: TEWPRec;
  IsIcon: Boolean;
begin
  Result := False;
  case Message.Msg of
    WM_WINDOWPOSCHANGED: begin
        IsIcon := IsIconic(Application.Handle);
        if IsMinimized <> IsIcon then begin
          IsMinimized := IsIcon;
          E.Window := Handle;
          E.NotMinimized := not IsMinimized and IsWindowVisible(Handle);
          EnumWindows (@EWP2, Longint(@E));
        end;
      end;
  end;
end;

procedure TMainForm.AppOnRestore (Sender: TObject);
var
  E: TEWPRec;
begin
  IsMinimized := False;
  E.Window := Handle;
  E.NotMinimized := True;
  EnumWindows (@EWP2, Longint(@E));
end;


procedure InitWindowsVersion;

  procedure ReadServicePackFromRegistry;
  var
    K: HKEY;
    Size, Typ, SP: DWORD;
  begin
    if RegOpenKeyEx(HKEY_LOCAL_MACHINE, 'System\CurrentControlSet\Control\Windows',
       0, KEY_QUERY_VALUE, K) = ERROR_SUCCESS then begin
      Size := SizeOf(SP);
      if (RegQueryValueEx(K, 'CSDVersion', nil, @Typ, @SP, @Size) = ERROR_SUCCESS) and
         (Typ = REG_DWORD) and (Size = SizeOf(SP)) then
        NTServicePackLevel := Word(SP);
      RegCloseKey (K);
    end;
  end;

type
  TOSVersionInfoEx = packed record
    dwOSVersionInfoSize: DWORD;
    dwMajorVersion: DWORD;
    dwMinorVersion: DWORD;
    dwBuildNumber: DWORD;
    dwPlatformId: DWORD;
    szCSDVersion: array[0..127] of Char;
    wServicePackMajor: Word;
    wServicePackMinor: Word;
    wSuiteMask: Word;
    wProductType: Byte;
    wReserved: Byte;
  end;
var
  OSVersionInfo: TOSVersionInfo;
  OSVersionInfoEx: TOSVersionInfoEx;
begin
  OSVersionInfo.dwOSVersionInfoSize := SizeOf(OSVersionInfo);
  if GetVersionEx(OSVersionInfo) then begin
    WindowsVersion := (Byte(OSVersionInfo.dwMajorVersion) shl 24) or
      (Byte(OSVersionInfo.dwMinorVersion) shl 16) or
      Word(OSVersionInfo.dwBuildNumber);
    { ^ Note: We MUST clip dwBuildNumber to 16 bits for Win9x compatibility }
    if IsNT then begin
      if OSVersionInfo.dwMajorVersion >= 5 then begin
        { OSVERSIONINFOEX is only available starting in Windows 2000 }
        OSVersionInfoEx.dwOSVersionInfoSize := SizeOf(OSVersionInfoEx);
        if GetVersionEx(POSVersionInfo(@OSVersionInfoEx)^) then
          NTServicePackLevel := (Byte(OSVersionInfoEx.wServicePackMajor) shl 8) or
            Byte(OSVersionInfoEx.wServicePackMinor);
      end
      else if OSVersionInfo.dwMajorVersion = 4 then begin
        { Read from the registry on NT 4 }
        ReadServicePackFromRegistry;
      end;
      { Note: NT 3.x doesn't have a DWORD registry entry for the service pack }
    end;
  end;
end;

procedure CreateEntryLists;
var
  I: TEntryType;
begin
  for I := Low(I) to High(I) do
    Entries[I] := TList.Create;
end;

procedure FreeEntryLists;
var
  I: TEntryType;
  J: Integer;
  L: TList;
begin
  for I := High(I) downto Low(I) do begin
    L := Entries[I];
    if Assigned(L) then begin
      for J := L.Count-1 downto 0 do begin
        if EntryStrings[I] <> 0 then
          SEFreeRec (L[J], EntryStrings[I])
        else
          FreeMem (L[J]);
        L[J] := nil;
      end;
      L.Free;
      Entries[I] := nil;
    end;
  end;
end;

initialization
  IsNT := UsingWinNT;
  InitWindowsVersion;
  NewGUI := WindowsVersion >= $04000000;
  InitComponents := TStringList.Create();
  WizardComponents := TStringList.Create();
  WizardTasks := TStringList.Create();
  CreateEntryLists;
  DeleteFilesAfterInstallList := TStringList.Create;
  DeleteDirsAfterInstallList := TStringList.Create;

finalization
  WizardImage.Free;
  WizardImage := nil;
  WizardSmallImage.Free;
  WizardSmallImage := nil;
  ScriptDeInit();

  DeleteDirsAfterInstallList.Free;
  DeleteFilesAfterInstallList.Free;
  FreeEntryLists;
  WizardTasks.Free();
  WizardComponents.Free();
  InitComponents.Free();
end.
